// This is a new portable back-end for Imp77, which uses GCC to create
// the object files from the imp source code.  It does this by outputing
// C rather than assembly code from the ICODE.  Note that this does not
// make it an imp to C translator - the C is not human-readable, nor is
// it maintainable.  It is merely a slightly higher level type of
// intermediate code.  However this will ensure that Imp code can be run
// on any system which has a GCC C compiler.

// Unfortunately, GCC is the required back-end compiler, because it has
// a non-standard extension that supports textually nested procedure
// calls, which none of the other portable C compilers do.  So for now,
// clang et al are not an option.  Although in principle the nested
// procedures can be exbedded and the code flattened, this is a major
// implementation project to handle, and the resulting C code is not
// going to be as efficient as code compiled for a language that uses
// a display to access enclosing scopes.  C's mechanism for nested
// procedure handling is made complicated by the need to support
// procedure variables, which Imp (and languages like Pascal, Algol,
// etc do not need).

// Note there is no need to perform optimisations at the AST level
// because GCC will do them for us.  Similarly things like folding
// const expressions or removing superfluous parentheses (although
// I did make an attempt at just that, with the --tidy option!)

// TO DO:
// * some pointer variables are not properly indirected when
//   evaluating expressions.  I did try putting a '*' in AST_VAR
//   but that wasn't quite right - it showed up in extra places
//   as well.  Needs to have a new AST_VALUE_OF tuple be created
//   and inserted only in the right places.  Also where I currently
//   am inserting a "&" before a variable, I need to change those
//   to use an AST_ADDRESS_OF tuple instead.

// * Procedure parameters:
//             2    %routine do(%integerfn fn(%integer p,q,r))
//     O LINE 2  OPD_STACK = 0  file=test/testprocparam.imp
//     $ DEF V_006f ID=DO a=7 b=0 c=0
//     { START (
//     $ DEF V_0070 ID=FN a=24 b=1 c=16
//     } FINISH )
//     $ DEF V_0070 ID=FN a=24 b=1 c=24
//     { START (
//     $ DEF V_0071 ID=P a=17 b=1 c=0
//     $ DEF V_0072 ID=Q a=17 b=1 c=0
//     $ DEF V_0073 ID=R a=17 b=1 c=0
//     } FINISH )

//   %switch middle,right,left(1:3)  - middle and right are not handled properly

// * Type casting for %map RECORD().  (Don't think imp77 supports %map ARRAY(addr, format)?
//   - instead of rec = *RECORD(addr) we should have rec = *(rec fm *)RECORD(addr)


// * potential precedence problem!:

  // ! test by:  ./i2c --stdout --tidy test/shiftadd.imp | fgrep "ATOM1 = "|head -1
  // %begin
  //   %integer atom1
  //  
  //   atom1 = (atom1-106)>>1+91;     !a,an->v  na,nan->n
  //  
  //   ! This minimal test seems to generate correct C but the same code in test/pass1.imp gives:
  //  
  //   ! //   1226                                          atom1 = (atom1-106)>>1+91;     !a,an->v  na,nan->n
  //   ! ATOM1 = (unsigned int)ATOM1 - 106 >> 1 + 91;
  //  
  //   ! reproduce it with: ./i2c --stdout --tidy test/pass1.imp | fgrep "ATOM1 = (unsigned int)ATOM1"
  //  
  // %endofprogram

// The "ATOM1 = (unsigned int)ATOM1 - 106 >> 1 + 91;" output is wrong because of
// C's precedence rules: it actually calculates: ATOM1 = ((unsigned int)ATOM1 - 106) >> (1 + 91);
// because + and - are higher priority than << and >>.  (2 vs 5, with 1 being hightest priority)

// which in turn causes this warning:
// test/pass1.c:2489:35: warning: right shift count >= width of type [-Wshift-count-overflow]
// ATOM1 = (unsigned int)ATOM1 - 106 >> 1 + 91;
//                                   ^~

// ###################################################################################################

/*
    This phase converts the ICODE into a more modern AST.  Each AST
   tuple also holds the type information of its object, so when an
   expression is constructed, the type information percolates upwards,
   being used for type conversion and widening as it goes.  The info
   could also be used for catching type errors but that won't be
   necessary here since the Imp front end has already made those checks.
     The type and form information is used primarily for handling
   automatic dereferencing of pointers and pointer functions - %name
   and %map in Imp terms.
     Note that the AST_* types rather grew on the fly as each statement
   type was gradually refined.  It's not the carefully designed AST
   that it should be, but it seems to be sufficient.  It does rely
   somewhat on the output of pass1 being in a particular format - some
   of the translation tricks would not work in the proper "imp to c"
   high-level translator project.
 */

#define _POSIX_C_SOURCE 200809L
#define _XOPEN_SOURCE 500
#include <string.h>

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <stdarg.h>

// wasn't worth a header file for one line...:
extern int pass1(char *source, char *perm, char *icode, char *list);

#include "i2c.h"
#include "ast.h"
#include "icode.h"

int PARM_C_IO = 0;  // output standard C I/O calls like printf rather than use IMP I/O names like PRINTSTRING
int PARM_IMP_SOURCE_LINENOS = 1;   // issue #line directives for better gdb error reporting
int PARM_LINK = 1;
int PARM_VERBOSE = 0;
int PARM_TIDY = 0;  // --tidy will try to reduce the number of parentheses in expressions
int PARM_STDOUT = 0;
int PARM_DEBUG = 0;
int PARM_OPT = 0;

// A couple of hacky globals to get me out of a tight spot...
static char *last_alias = NULL;
static ASTIDX last_declare_for_init = -1;
static ASTIDX last_declare_for_attaching_fields_or_parameters = -1;
static int IN_START_FINISH_GROUP = 0;

// Hack for %switch left,middle,right(low:high) ...
static ASTIDX last_bounds_pair = -1;

extern short get_next_label_id(void);
extern int NEXT_AST;

//#define MAX_NESTED_BLOCKS 1024 // More than Imp ever allowed...
int /*StringpoolIDX*/ blockname[MAX_NESTED_BLOCKS];
int blocklevel = 0;

int want_extra_info = FALSE;
int suppress_perms = TRUE;
int in_perms = TRUE;
int suppress_icode = TRUE;

FILE *icode_file;
FILE *output_file;
FILE *source_file;

#define MAX_LINE 1024
char line[MAX_LINE + 1];

// Diagnostic check to remind me of unimplemented calls.
static int unimplemented[256];  // (Not really unimplemented, more a case of not having been
                                // seen in a test, so potentially causing a stack imbalance)

void COPY_FIXED_FIELDS_to_from(ASTIDX new, ASTIDX old) {
  // defaults, but need to be overridden depending on the operation
  // **** NOTE **** userfields and extended fields are NOT copied over by default!
  // This is primarily for propogating type info.  I can probably avoid copying
  // a bunch of the fields.

  BASE_TYPE(new) = BASE_TYPE(old);  // T_*  eg T_INTEGER, T_REAL
  FORM(new) = FORM(old);            // F_*  eg F_FN
  BASE_SIZE_CODE(new) = BASE_SIZE_CODE(old);
  BASE_SIZE_BYTES(new) = BASE_SIZE_BYTES(old);

  LINKAGE(new) = LINKAGE(old);  // X_*  eg X_OWN, X_CONST etc
  SPECIAL(new) = SPECIAL(old);  // SPECIAL_*

  IMP_NAME_IDX(new) = IMP_NAME_IDX(old);
  C_NAME_IDX(new) = C_NAME_IDX(old);
  EXTERNAL_NAME_IDX(new) = EXTERNAL_NAME_IDX(old);

  IS_BASE_NAME(new) = IS_BASE_NAME(old);
  IS_ARRAY(new) = IS_ARRAY(old);
  IS_ARRAY_NAME(new) = IS_ARRAY_NAME(old);

  IS_SPEC_ONLY(new) = IS_SPEC_ONLY(old);    // { 8,  " spec" }, // S=1
  NO_AUTO_DEREF(new) = NO_AUTO_DEREF(old);  // { 16, " {indirect-no-auto-deref, I=1}" }
  NO_UNASS(new) = NO_UNASS(old);            // { 32, " {NO UNASSIGNED CHECKS, U=1}" }

  STRING_CAPACITY(new) = STRING_CAPACITY(old);
  RECORD_FORMAT(new) = RECORD_FORMAT(old);
  FORMAL_PARAM_LIST(new) = FORMAL_PARAM_LIST(old);
  ACTUAL_PARAM_LIST(new) = ACTUAL_PARAM_LIST(old);
  BOUNDS1D(new) = BOUNDS1D(old);
  DOPEVECTOR(new) = DOPEVECTOR(old);
  INITVALUES(new) = INITVALUES(old);  // index into a InitValues[] array of AST indexes.  See INIT code.
}

#ifdef NEVER

// This was for testing temporaries to use in double-sided conditions.  It should
// be possible, but this version did not work.  Same mechanism is also going to
// be needed for freezing the increment and final values of for loops.  And
// possibly elsewhere.  Correcting the order of evaluation of parameters for
// example.  Note even in double-sided conditions we need to be careful about
// the order of evaluation of each of the 3 elements if they can have side-effects.

ASTIDX get_temp_for(ASTIDX template) {
  // Generate a temporary var that 'template' can be assigned to.
  int tag = NEW_INTERNAL_TAG();
  ASTIDX tmp = mktuple(AST_DECLARE, tag);
  C_NAME_IDX(tmp) = strtopool("tmp");
  COPY_FIXED_FIELDS_to_from(tmp, template);
  PUSH(tmp);
  ASTIDX var = mktuple(AST_VAR, tag);
  return var;
}
#endif

void FLUSH_ICODE_STACK(void) {
  // By the time this is called, every item on the stack is a top-level source statement, and
  // the entire stack, if popped and executed, would generate code in reverse order!

  // However because each item on the stack is a complete statement, it's actually safe for
  // us to cycle through the stack and execute each AST item in sequential order!

  if (suppress_perms && in_perms) {
    // don't.
  } else {
    fprintf(output_file, "%s", commentbuff);
  }
  cbp = commentbuff;

  for (int sp = 0; sp < next_opd; sp++) {
    ASTIDX code = opdstack[sp];
    if (OP(code) != AST_IMP_LINE) fprintf(output_file, "%s", hashline);  // #line set up by LINE ('O') opcode.
    codegen(code);
  }
  next_opd = 0;  // empty the stack!
}

int sysnprintf(char *out, int maxlen, char *command) {
  FILE *pipe;
  pipe = popen(command, "r");
  for (int i = 0; i < maxlen - 1; i++) {
    int c = fgetc(pipe);
    if (c == EOF) {
      out[i] = '\0';
      pclose(pipe);
      return i;
    }
    out[i] = c;
  }
  return 0;
}

int main(int argc, char **argv) {
  //char *p1, *p2;
  int opcode, d;
  char list_filename[256], icode_filename[256], output_filename[256], source_filename[256], object_filename[256],
      base_filename[256], perm_filename[256];

  char *PROGNAME = argv[0];
  object_filename[0] = '\0';

  sprintf(perm_filename, "perms.inc");

  while (argc >= 2 && *argv[1] == '-') {
    // options must come before the filename

    // TO DO: -h/--help

    if (argc >= 2 && (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)) {
      fprintf(stderr, "syntax: i2c [options] file.imp\n");
      fprintf(stderr, "\n");
      fprintf(stderr, "Options:\n");
      fprintf(stderr, "    --verbose/-v       Show AST tuple generation\n");
      fprintf(stderr, "    --debug/-d         Debug icode stack usage\n");
      fprintf(stderr, "    -O                 Optimise run-time performance.  Default is full checks\n");
      fprintf(stderr, "    --icode            Decode the intermediate code to the C file as comments\n");
      fprintf(stderr, "    -c                 Compile to .o file only. Don't link.\n");
      fprintf(stderr, "    --stdout           Output the C code to stdout instead of a .c file\n");
      fprintf(stderr, "    -o binary[.o]      When generating a binary use this filename\n");
      fprintf(stderr, "    --tidy             Output C code with fewer redundant brackets\n");
      fprintf(stderr, "    --perms perm.inc   Use alternative perms.inc file\n");
      fprintf(stderr, "    --no-perms         Use the empty \"noperms.inc\" file.\n");
      fprintf(stderr, "    --show-perms       Print the translated C of the perms.inc file\n");
      fprintf(stderr, "    --[no-]gdb         Generate #line statements in the C\n");
      fprintf(stderr, "    --[no-]check       Add runtime checking.  Small runtime overhead.\n");
      fprintf(stderr, "                       (Code runs under valgrind when checks are enabled.)\n");
      fprintf(stderr, "\n");
      exit(0);
    } else if (argc >= 2 && (strcmp(argv[1], "-v") == 0 || strcmp(argv[1], "--verbose") == 0)) {
      PARM_VERBOSE = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && (strcmp(argv[1], "-d") == 0 || strcmp(argv[1], "--debug") == 0)) {
      debug_ast = PARM_DEBUG = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "-c") == 0) {
      PARM_LINK = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "-O") == 0) {
      PARM_OPT = TRUE;
      PARM_IMP_SOURCE_LINENOS = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 3 && strcmp(argv[1], "--perms") == 0) {
      if (argv[2] == NULL || argv[2][0] == '-') {
        fprintf(stderr, "i2c: --perms option requires a filename parameter\n");
        exit(1);
      }
      strcpy(perm_filename, argv[2]);
      argc -= 2;
      argv += 2;
    } else if (argc >= 3 && strcmp(argv[1], "-o") == 0) {
      if (argv[2] == NULL || argv[2][0] == '-') {
        fprintf(stderr, "i2c: -o option requires a filename parameter\n");
        exit(1);
      }
      strcpy(object_filename, argv[2]);
      argc -= 2;
      argv += 2;
    } else if (argc >= 2 && (strcmp(argv[1], "--no-perms") == 0 || strcmp(argv[1], "--noperms") == 0)) {
      strcpy(perm_filename, "noperms.inc");
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--stdout") == 0) {
      PARM_STDOUT = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--line") == 0) {
      PARM_IMP_SOURCE_LINENOS = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--no-line") == 0) {
      PARM_IMP_SOURCE_LINENOS = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--check") == 0) {
      // enable runtime checks with overhead, eg unassigned checks
      // and use valgrind etc at runtime.
      PARM_OPT = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--no-check") == 0) {
      PARM_OPT = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--icode") == 0) {
      suppress_icode = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--tidy") == 0) {
      PARM_TIDY = TRUE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && strcmp(argv[1], "--show-perms") == 0) {
      suppress_perms = FALSE;
      argc -= 1;
      argv += 1;
    } else if (argc >= 2 && argv[1][0] == '-') {
      fprintf(stderr, "i2c: unrecognised option '%s'\n", argv[1]);
      exit(EXIT_FAILURE);
    }
  }

  if (argc != 2) {
    fprintf(stderr, "syntax: i2c [options] file.imp\n");
    exit(1);
  }

  strcpy(base_filename, argv[1]);
  if (strcmp(&base_filename[strlen(base_filename) - 4], ".imp") == 0) {
    base_filename[strlen(base_filename) - 4] = '\0';
  }

  sprintf(source_filename, "%.250s.imp", base_filename);
  if (PARM_STDOUT) {
    sprintf(output_filename, "/dev/stdout");
  } else {
    sprintf(output_filename, "%.250s.c", base_filename);
  }
  sprintf(icode_filename, "%.250s.icd", base_filename);
  sprintf(list_filename, "%.250s.lis", base_filename);
  int rc = pass1(source_filename, perm_filename, icode_filename, list_filename);
  (void)rc;

  output_file = fopen(output_filename, "w");
  if (output_file == NULL) {
    fprintf(stderr, "i2c: cannot create '%s' - %s\n", output_filename, strerror(errno));
    exit(2);
  }

  source_file = fopen(source_filename, "r");
  if (source_file == NULL) {
    fprintf(stderr, "i2c: Cannot open \"%s\" - %s\n", source_filename, strerror(errno));
    exit(3);
  } else {
    dump_comment("%s\n\n\n", "Edinburgh IMP77 Compiler - Version 8.4\n");
  }

  icode_file = fopen(icode_filename, "rb");
  if (icode_file == NULL) {
    fprintf(stderr, "i2c: cannot open '%s' - %s\n", icode_filename, strerror(errno));
    exit(4);
  }

  ASTIDX NULL_TUPLE = mktuple(AST_ASTCODE_WAS_ZERO);  // for catching unassigned variables...
  short NULL_LABEL = get_next_label_id();
  blockname[0] = strtopool("__LEVEL_0__");

  dump_code("#ifdef USE_PERMS_INC\n");
  for (;;) {
    opcode = get_icode(icode_file);
    if (opcode == EOF) break;
    switch (opcode) {
      case '\n':
        break;  // There's a \n at the end of the icode file...
        //case '\r': break; // Shouldn't get this but there might be one in a Windows port?

      case ':': /*  Define Compiler Label(Tag); */ /* LOCATE label */
      {
        int lab, labtag = getshort();

        //lab = (label_id[labtag] != 0 ? setforwardlab_from_tag(labtag) : setbackwardlab_from_tag(labtag));

        if (label_id[labtag] == 0) {
          lab = label_id[labtag] = get_next_label_id();
          dump_comment("%c %s  (labtag=%04x lab=L_%04x)\n", opcode, icode_name[opcode], labtag, lab);
          PUSH(mktuple(AST_DEFLAB, mktuple(AST_LABEL, lab, 'L', labtag)));  // start of cycle
          label_id[labtag + 1] = get_next_label_id();
          label_id[labtag + 2] = get_next_label_id();
        } else {
          lab = label_id[labtag];
          PUSH(mktuple(AST_DEFLAB, mktuple(AST_LABEL, lab, 'L', labtag)));  // continue or exit label of cycle
          dump_comment("%c %s  (labtag=%04x lab=L_%04x)\n", opcode, icode_name[opcode], labtag, lab);
          label_id[labtag] = 0;
        }

        // These will be freed up on the REPEAT opcode.

      } break;

      case 'B': /*  Jump Backward(Tag); */ /* REPEAT label */
      {
        int labtag = getshort();
        int forlab = label_id[labtag];
        dump_comment("%c %s  (tag=%04x, lab=L_%04x)\n", opcode, icode_name[opcode], labtag, forlab);
        PUSH(mktuple(AST_GOTO, mktuple(AST_LABEL, forlab, 'L', labtag)));
        // After a REPEAT, that label tag will not be used again to represent
        // that same label:
        label_id[labtag] = 0;  // free it up for reuse.
      } break;

      case 'f': /*  Compile For(Tag); */ /* FOR label (label was missing from thesis description) */
      {
        /*
               The descriptors on the top of the stack are:-
         
                                  +-----+
                                  |     |
               INITIAL VALUE      | [F] |
                                  |     |
                                  |-----|
                                  |     |
               FINAL VALUE        | [L] |
                                  |     |
                                  |-----|
                                  |     |
               INCREMENT          | [I] |
                                  |     |
                                  |-----|
                                  |     |
               CONTROL VARIABLE   | [C] |
                                  |     |
                                  |-----|
                                  |     |
                                  .     .


                    *** NOTE ***  TO DO: Final should be evaluated once and saved in a temporary
                                  which is to be used instead, if final can potentially have side-effects,
                                  i.e. the terminating value is the value of final on entry to the
                                  for loop - it is not to be recalculated on every loop.

                 */

        short int fortag;
        short int forlab;
        short int exitlab;
        short int continuelab;

        fortag = gettag();
        forlab = label_id[fortag] = get_next_label_id();
        exitlab = label_id[fortag + 1] = get_next_label_id();
        continuelab = label_id[fortag + 2] = get_next_label_id();
        (void)continuelab;

        // forlab is at the top of the loop, continuelab is just before the repeat
        // (where a %until could possibly be tested). exitlab is after the loop

        //dump_comment("// FOR tag:lab  %d:%d  %d:%d  %d:%d\n", fortag,forlab, fortag+1,exitlab, fortag+2,continuelab);

        dump_comment("%c %s L_%04x\n", opcode, icode_name[opcode], forlab);

        ASTIDX initial = POP();
        ASTIDX final = POP();
        ASTIDX increment = POP();
        ASTIDX control = POP();

        // Could optimise but as a start this basic logic will suffice:

        // <control> = <initial> - <increment>
        ASTIDX stmnt1 =
            mktuple(AST_ASSIGN, 'S', control, mktuple(AST_BINOP, '-', initial, increment));  // order for '-'?

        // LABEL:
        ASTIDX stmnt2 = mktuple(AST_DEFLAB, mktuple(AST_LABEL, forlab, 'L', fortag));

        // if (<control> == <final>) goto <exit label>
        ASTIDX done = mktuple(AST_COMPARE, control, '=', final);         // TO DO: swap order here and in ast.c
        ASTIDX exit_ast = mktuple(AST_LABEL, exitlab, 'L', fortag + 1);  // try (and fail) to generate exit label
        ASTIDX stmnt3 = mktuple(AST_IFGOTO, done, exit_ast);

        // <control> = <control> + <increment>
        ASTIDX stmnt4 = mktuple(AST_ASSIGN, 'S', control, mktuple(AST_BINOP, '+', control, increment));

        // Ensure all the above are executed in sequence!
        PUSH(mktuple(AST_SEQ, stmnt1, mktuple(AST_SEQ, stmnt2, mktuple(AST_SEQ, stmnt3, stmnt4))));
      } break;

      case 'F': /*  Jump Forward(Tag, Always); */ /* GOTO label */
      {
        short int labtag;
        short int lab = getforwardlab_from_tag(&labtag);

        dump_comment("%c %s (tag=%04x, lab=L_%04x)\n", opcode, icode_name[opcode], labtag, lab);
        PUSH(mktuple(AST_GOTO, mktuple(AST_LABEL, lab, 'L', labtag)));
      } break;

      case 'L': /*  Define User Label(Tag); */ /* LABEL label */
      {
        int ulab = gettag();
        // user labels are described by tags, can be stacked as params!
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_DEFLAB, mktuple(AST_LABEL, ulab, 'U', ulab)));
        // TODO AST
      } break;

      case 'J': /*  User Jump(Tag); */ /* JUMP label */
      {
        int ulab = gettag();
        // user labels are described by tags, can be stacked as params!
        dump_comment("%c %s (lab=U_%04x)\n", opcode, icode_name[opcode], ulab);
        PUSH(mktuple(AST_GOTO, mktuple(AST_LABEL, ulab, 'U', ulab)));
      } break;

      case '"': /*  Compare Double; */ /* FORWARD-JUMPIFD cond label */
      {
        // NOTE!!! 'Compare double' refers to double-sided condition,
        //  not to C's double reals

        /* leaves tos on the stack */

        /* TO DO: if the middle value can potentially have side-effects,
                          e.g. in %if 'A' <= getsymbol <= 'Z' ... then it should
                          be saved to a temporary when it is first evaluated and
                          that temporary used in the second test.  Also if the
                          first value can have side-effects remember that in Imp
                          they are evaluated left to right so you cannot simply
                          load the middle value first, in case some idiot has
                          written code akin to %if getsymbol <= getsymbol <= 999 ...
                          The C code we need to generate is:

                              if ( ({int tmp1 = a, tmp2 = b, tmp3 = c; tmp1 < tmp2 && tmp2 < tmp3;}) ) goto <lab>;


         */

        int cond = getbyte();
        short int labtag;
        short int lab = getforwardlab_from_tag(&labtag);
        dump_comment("%c %s %c L_%04x\n", opcode, icode_name[opcode], cond, lab);

        ASTIDX tos = POP();
        ASTIDX sos = POP();

        // ASTIDX temp = get_temp_for(sos);

        ASTIDX comparison = mktuple(AST_COMPARE, sos, cond, tos);  // TO DO: swap order here and in ast.c
        PUSH(mktuple(AST_IFGOTO, comparison, mktuple(AST_LABEL, lab, 'L', labtag)));
        PUSH(tos);
      } break;

      case 'C': /*  Compare Addresses; */ /* FORWARD-JUMPIFA cond label */
      case '?': /*  Compare Values; */    /* FORWARD-JUMPIF cond label */
        /* if you have the JUMPIF first then the LOCATE clears it. */
        /* if you have the LOCATE first then the REPEAT clears it. */
        {
          int cond = getbyte();
          short int labtag;
          short int lab = getforwardlab_from_tag(&labtag);
          dump_comment("%c %s %c L_%04x\n", opcode, icode_name[opcode], cond, lab);

          ASTIDX tos = POP();
          ASTIDX sos = POP();

          ASTIDX comparison = mktuple(AST_COMPARE, sos, cond, tos);  // TO DO: swap order here and in ast.c
          PUSH(mktuple(AST_IFGOTO, comparison, mktuple(AST_LABEL, lab, 'L', labtag)));
        }
        break;

      case 't': /*  Jump Forward(Tag, TT); */ /* BT label */
      case 'k': /*  Jump Forward(Tag, FF); */ /* BF label */
      {
        // This might be for C/Pascal booleans and not used in Imp where we
        // have short-cut evaluation of conditions,  and if so, it may need
        // a boolean to be popped off the compiler stack.
        // I.e. this implements if (cond) goto lab, and if (!cond) goto lab.

        // Oh - wait - they're also for testing the results of string resolution, and
        // also %predicate results (%true, %false).  Need to create tests for those.
        short int labtag;
        short int lab = getforwardlab_from_tag(&labtag);
        dump_comment("%c %s L_%04x\n", opcode, icode_name[opcode], lab);
        ASTIDX condition = POP();
        PUSH(mktuple(AST_IFGOTO, condition, mktuple(AST_LABEL, lab, 'L', labtag)));
      } break;

      case '$': /*  Define Var; */ /* DEF TAG TEXT TYPE FORM SIZE SPEC PREFIX */
      {
        char *alias = last_alias;  // now we have it, still need to do something with it...
        last_alias = NULL;

        // Instruction:  Define <tag> [id] <a> <b> <c>
        int tag;
        char *ID;
        short int a, b, cX, i;

        tag = gettag();
        ID = getname();
        assert(get_icode(icode_file) == ',');
        a = gettag() & 0xFFFF;
        assert(get_icode(icode_file) == ',');
        b = gettag() & 0xFFFF;
        assert(get_icode(icode_file) == ',');
        cX = gettag() & 0xFFFF;

        if (tag == 0) {
          tag = NEW_INTERNAL_TAG();  // Invented tag so that record fields can be looked up to handle SELECT
          // We should be able to use the extended field option in the AST to store these.
          // Record fields are numbered from 1 in SELECT.
        }
        dump_comment("%c %s %c_%04x ID=%s a=%d b=%d c=%d  \n", opcode, icode_name[opcode],
                     tag >= next_free_private_tag ? 'S' : 'V',  // 'system' tag
                     tag, ID, a, b, cX);

        ASTIDX decl = mktuple(AST_DECLARE, tag);
        last_declare_for_init = decl;  // use the other hacky global :-/
        Descriptor[tag] = decl;        // tag_declaration;

        //
        // Currently when outputting %result == mapresult (where mapresult is a %integermap), codegen is seeing FORM=0
        // rather than FORM=9 (F_MAP) as expected.  Trying to find out why...
        //
        // $ DEF V_0071 ID=MAPRESULT a=25 b=1 c=0   <--- 25&15 == 9
        //
        BASE_TYPE(decl) = (a >> 4) & 0xF;
        FORM(decl) = a & 0xF;

        /* handle bizarre type encoding scheme: */
        SPECIAL(decl) = 0;
        if (BASE_TYPE(decl) == T_INTEGER && b == 2)
          SPECIAL(decl) = 1; /* integer-oid */
        else if (BASE_TYPE(decl) == T_INTEGER && b == 3)
          SPECIAL(decl) = 2; /* integer-oid */
        else if (BASE_TYPE(decl) == T_REAL && b == 4)
          SPECIAL(decl) = 3;

        LINKAGE(decl) = cX & 7;

        if (FORM(decl) == F_ARRAY || FORM(decl) == F_NAME_ARRAY) IS_ARRAY(decl) = TRUE;
        if (FORM(decl) == F_NAME || FORM(decl) == F_NAME_ARRAY || FORM(decl) == F_NAME_ARRAY_NAME)
          IS_BASE_NAME(decl) = TRUE;
        if (FORM(decl) == F_ARRAY_NAME || FORM(decl) == F_NAME_ARRAY_NAME) IS_ARRAY_NAME(decl) = TRUE;

        if (SPECIAL(decl)) {
        } else if (BASE_TYPE(decl) == T_STRING) {
          if (b == 0) {
            STRING_CAPACITY(decl) =
                -1;  // '*'.  Maybe should use 0 since %string(0) fred is pretty stupid and probably caught by pass1?
          } else {
            STRING_CAPACITY(decl) = b;
          }
        } else if (BASE_TYPE(decl) == T_INTEGER) {
          const int datasize[4] = {8, 4, 1, 2};
          BASE_SIZE_CODE(decl) = b & 3;
          BASE_SIZE_BYTES(decl) = datasize[BASE_SIZE_CODE(decl)];
        } else if (BASE_TYPE(decl) == T_REAL) {
          const int datasize[8] = {4, 4, 4, 4, 8, 4, 4, 4};
          BASE_SIZE_CODE(decl) = b & 7;  // NOTE: 8 values for reals, 4 for integers!  BE CAREFUL!
          BASE_SIZE_BYTES(decl) = datasize[BASE_SIZE_CODE(decl)];
        } else if (BASE_TYPE(decl) == T_RECORD) {
          if (b == 0) {
            RECORD_FORMAT(decl) = 0;  // %const %record(*) %name nil == 0
          } else {
            RECORD_FORMAT(decl) = Descriptor[b];
            //debug_types(decl);
          }
        }

        for (i = 0; i < sizeof(oflags) / sizeof(oflags[0]); i++) {
          if (cX & oflags[i].mask) {
            if (oflags[i].mask == 8) IS_SPEC_ONLY(decl) = TRUE;    // { 8,  " spec" }, // S=1
            if (oflags[i].mask == 16) NO_AUTO_DEREF(decl) = TRUE;  // { 16, " {indirect-no-auto-deref, I=1}" }
            if (oflags[i].mask == 32) NO_UNASS(decl) = TRUE;       // { 32, " {NO UNASSIGNED CHECKS, U=1}" }
          }
        }

        EXTERNAL_NAME_IDX(decl) = C_NAME_IDX(decl) = IMP_NAME_IDX(decl) = strtopool(ID);

        if (FORM(decl) == F_MAP) {
          // Needs to be marked so that if assigning to a scalar with '=', it isindirected through '*'
          // but not if assigning to a pointer with '=='
          // Test minmap.imp (minimal map test) is not yet working.
        }

        if (alias != NULL) EXTERNAL_NAME_IDX(decl) = strtopool(alias);

        // Now handle declarations which have other parameters already on the internal stack:

        if (FORM(decl) == F_SWITCH) {
          /*

            Multiple switch declarations with only the last being given the bounds are not being handled
            properly.  This construct works OK for arrays so need to use or copy that mechanism for switches too:

                        // O LINE 3  OPD_STACK = 0  file=test/test164a.imp
                        // N PUSHI #1  OPD_STACK = 2
                        // N PUSHI #3  OPD_STACK = 3
                        // b BOUNDS
                        // $ DEF V_0074 ID=MIDDLE a=6 b=0 c=0
                        // $ DEF V_0075 ID=RIGHT a=6 b=0 c=0
                        // $ DEF V_0076 ID=LEFT a=6 b=0 c=0


              //      3      %switch middle,right,left(1:3)
            static const void * / *SWITCH* / LEFT
            static const void * / *SWITCH* / RIGHT
            static const void * / *SWITCH* / MIDDLE[(3)-(1)+1] = { &&MIDDLE_1, &&MIDDLE_2, &&MIDDLE_3,  };
               = {  };
               = {  };


           Nope... that syntax for arrays only works for auto arrays (potentially n-dimensional), not
           for %own or %external arrays etc with fixed 1-D bounds!

                        // O LINE 3  OPD_STACK = 0  file=test/threearrays.imp
                        // $ DEF V_0074 ID=LEFT a=27 b=1 c=256
                        // Auto-array decl 13163 requires following DIM for bounds
                        // $ DEF V_0075 ID=MIDDLE a=27 b=1 c=256
                        // Auto-array decl 13192 requires following DIM for bounds
                        // $ DEF V_0076 ID=RIGHT a=27 b=1 c=256
                        // Auto-array decl 13221 requires following DIM for bounds
                        // N PUSHI #20  OPD_STACK = 5
                        // N PUSHI #20  OPD_STACK = 6
                        // d DIM 1 3
                        // creating a 1D array dopevector
                        // Dimension 0: low 13250 high 13279
                        // assigning a dopevector 13338 to array declaration 13221
                        // assigning a dopevector 13368 to array declaration 13192
                        // assigning a dopevector 13398 to array declaration 13163
                                                                //      2
                                                                //      3    %integerarray left, middle, right(20:20)
int LEFT[(20)-(20)+1];
int MIDDLE[(20)-(20)+1];
int RIGHT[(20)-(20)+1];

           So it looks like a special hack is going to be needed to remember the last bounds pair handled.  *TO DO*

           */
          
          ASTIDX bounds = POP();
          if (OP(bounds) != AST_BOUNDSPAIR) {
            if (last_bounds_pair == -1) {
              fprintf(stderr, "Cannot determine bounds for %%switch %s\n", pooltostr(C_NAME_IDX(decl)));
              exit(1);
            }
            BOUNDS1D(decl) = bounds = last_bounds_pair;
          } else {
            BOUNDS1D(decl) = bounds;
          }
          
          last_bounds_pair = bounds;

          // keep an array of switch indices that have been defined
          // for this switch variable, and record this switch variable on a
          // stack at the current block level.  When we reach the end of the
          // current block level, output the code to handle any undefined
          // switch labels.

          ASTIDX lowidx, highidx;
          int low, high;
          ASTIDX AST_check_op;
          detuple(bounds, &AST_check_op, &lowidx, &highidx);
          detuple(lowidx, &AST_check_op, &low);
          detuple(highidx, &AST_check_op, &high);
        
          // space to record array of flags(low:high)
          switches_set[swstack_nextfree] = (char *)calloc((high-low+1), sizeof(char));
          switch_low[swstack_nextfree] = low;
          switch_high[swstack_nextfree] = high;
          switch_tag[swstack_nextfree] = tag;
          //switch_decl[swstack_nextfree] = decl;
          swstack_nextfree += 1;
          
        } else if (IS_ARRAY(decl)) {
          // NOTE!!! Bounds for auto arrays are added afterwards in a DIM code,
          // but bounds for own or external arrays (which must be constant and
          // 1 dimensional) are below the $DEF on the stack.
          if (LINKAGE(decl) == X_AUTO) {
            dump_comment("Auto-array decl %d requires following DIM for bounds\n", decl);
          } else {
            // Pick up 1D arraybounds and attach to decl
            ASTIDX bounds = POP();
            dump_comment("Attaching AST_BOUNDSPAIR %d to decl %d\n", bounds, decl);
            BOUNDS1D(decl) = bounds;
          }

        } else if (IS_PROCEDURE(decl)) {  // callable?

          if (!IN_START_FINISH_GROUP) last_declare_for_attaching_fields_or_parameters = decl;

        } else if (FORM(decl) == F_RECORDFORMAT) {
          if (!IN_START_FINISH_GROUP) last_declare_for_attaching_fields_or_parameters = decl;

        } else {
          // MANY MORE TO BE HANDLED.  RECORD. ARRAYS. ETC.
        }

        if (IN_START_FINISH_GROUP) {
          // This is actually a recordformat field declaration, so has to
          // be attached to the currently active recordformat.  (Not sure
          // yet about record subfields within a record format. **CHECK**

          // These declarations are pushed back on the stack and are gathered
          // up when the ALTEND code is issued, which will collect all the
          // record fields between ALTBEG and ALTEND off the stack, and then
          // attach them to the RECORDFORMAT definition which will have been
          // immediately below them on the stack.

          // NOTE: The ALT structure of a recordformat only has to be preserved
          // until we've output the declaration.  It's not actually needed when
          // accessing a field with SELECT which basically just passes the
          // field name and type info to C and which doesn't care that there
          // are other variables sharing the same space.  So - although I
          // haven't worked out the details yet - I think we'll be able to
          // get away with a relatively cheap hack.  At least I hope so because
          // I really don't have any infrastructure in place to handle
          // struct unions at this point :-(

          PUSH(decl);  // if (PARM_VERBOSE) debug_types(decl);
        } else {
          PUSH(decl);  // if (PARM_VERBOSE) debug_types(decl);

          // Making a change to the AST handling: The decl just prints
          // the procedure heading - the start of the procedure body
          // will come from a second AST node that is a blockstart
          if (IS_PROCEDURE(decl) && !IS_SPEC_ONLY(decl)) {
            // Start of a rt/fn/pred/map causing a new blocklevel:
            swbase[blocklevel] = swstack_nextfree;
            int BLOCKTYPE = 0;
            if (FORM(decl) == F_ROUTINE) BLOCKTYPE = BLOCKTYPE_ROUTINE;
            else if (FORM(decl) == F_FN) BLOCKTYPE = BLOCKTYPE_FN;
            else if (FORM(decl) == F_MAP) BLOCKTYPE = BLOCKTYPE_MAP;
            else if (FORM(decl) == F_PREDICATE) BLOCKTYPE = BLOCKTYPE_PREDICATE;
            PUSH(mktuple(AST_BLOCKSTART, blocklevel+1, BLOCKTYPE, C_NAME_IDX(decl)));
          }
          
        }
      } break;

      case '(':                               /*  Jump Forward(Tag, LE); */
      case ')':                               /*  Jump Forward(Tag, GE); */
      case '<':                               /*  Jump Forward(Tag, LT); */
      case '>':                               /*  Jump Forward(Tag, GT); */
      case '=':                               /*  Jump Forward(Tag, EQ); */
      case '#': /*  Jump Forward(Tag, NE); */ /* BNE */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], getshort());
        break;

      case 'W': /*  Switch Jump(Tag); */ /* SJUMP sd */
      {
        // The (constant) index of this switch jump is on the internal stack.
        int tag = gettag();
        dump_comment("%c %s V_%04x\n", opcode, icode_name[opcode], tag);
        ASTIDX index_ast = POP();
        ASTIDX sw = Descriptor[tag];
        PUSH(mktuple(AST_GOTO_SWLAB, sw, index_ast));

      } break;

      case '_': /*  Switch Label(Tag); */ /* SLABEL sd */
      {
        // ***NOTE*** If a sw(*) is given, pass1 generates all the missing labels.
        //            *but* if no sw(*) is present, no extra labels are generated
        //            and no explicit icode is issued either.  So when the switch
        //            label was declared, it's up to the icode handler to pre-fill
        //            the array with a label that points to code which will issue
        //            a run-time error message! (Which will have to be created)

        // The (constant) index of this switch label is on the internal stack.
        int tag = gettag();
        ASTIDX const_switch_index_ast = POP();  // CHECKPOP(); // tag or const value???
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], tag);

        PUSH(mktuple(AST_DEF_SWLAB, mktuple(AST_VAR, tag), const_switch_index_ast));

        // record the existence of this switch label to help with handling
        // defaults that are to be dumped at the end of a block.

        // locate the switch definition in the stack...
        int swstack_idx = swstack_nextfree;
        for (;;) {
          swstack_idx -= 1;
          if (swstack_idx < 0) break;
          if (switch_tag[swstack_idx] == tag) {
            int low = switch_low[swstack_idx];
            //int high = switch_high[swstack_idx];
            int switch_index = USERFIELD(const_switch_index_ast, 0);
            char *switch_flag = switches_set[swstack_idx];
            if (low >= 1234000) {
              dump_code("/*BUG: low=%d for swstack_idx=%d*/", low, swstack_idx);
            } else {
              switch_flag[switch_index-low] = 1; // MARK AS USED!
            }
            break;
          }
        }
        
      } break;

      case 'Z': /* ASSREF */ /*  Assign(EqualsEquals); */
      case 'j': /* JAM */    /*  Assign(Jam); */
      case 'S': /* ASSVAL */ /*  Assign(Equals); */
      {
        ASTIDX tos = POP();
        ASTIDX sos = POP();
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_ASSIGN, opcode, sos, tos /*order?*/));  // and push
      } break;

      case '!': /* OR */                               /*  Operation(ORx); */
      case '%': /* XOR */                              /*  Operation(XORx); */
      case '&': /* AND */                              /*  Operation(ANDx); */
      case '.': /* CONCAT */                           /*  Operation(CONCx); */
      case '+': /* ADD */                              /*  Operation(ADDx); */
      case '*': /* MUL */                              /*  Operation(MULx); */
      case '-': /* SUB */                              /*  Operation(SUBx); */
      case 'Q': /* DIVIDE - real divide */             /*  Operation(RDIVx); */
      case '/': /* QUOT - integer divide */            /*  Operation(DIVx); */
      case '[': /* LSH */                              /*  Operation(LSHx); */
      case ']': /* RSH */                              /*  Operation(RSHx); */
      case 'X': /* IEXP */                             /*  Operation(EXPx); */
      case 'x': /* REXP */                             /*  Operation(REXPx); */
      {
        ASTIDX tos = POP();
        ASTIDX sos = POP();
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_BINOP, opcode, sos, tos /*order?*/));  // and push
      } break;

      case 'v': /* MOD */ /*  Operate(ABSx, Stack, Nil); */
        // Depending on the type of the operand, call one of:
        //        double fabs(double x);
        //        float fabsf(float x);
        //        long double fabsl(long double x);

      case '\\': /* NOT */   /*  Operate(NOTx, Stack, Nil); */
      case 'U': /* NEGATE */ /*  Operate(NEGx, Stack, Nil); */
      {
        ASTIDX tos = POP();
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_MONOP, opcode, tos));  // and push
      } break;

      case 'H': /* BEGIN */ /*  Compile Begin; */
      {
        int BLOCKTYPE;
        static int unique_blockno = 0;
        char tmpname[128];
        // nested scope.  Important if we're using imp names for variables.
        //dump_comment("%c %s  blocklevel=%d before incrementing\n", opcode, icode_name[opcode], blocklevel);
        dump_comment("%c %s\n", opcode, icode_name[opcode]);

        if (blocklevel == 0) { // a %begin at the outer level... either %begin/%endofprogram or %begin/%end/%endoffile (which is identical)
          BLOCKTYPE = BLOCKTYPE_BEGIN_ENDOFPROGRAM;
          sprintf(tmpname, "_imp_main");
        } else {
          BLOCKTYPE = BLOCKTYPE_BEGIN_END;
          sprintf(tmpname, "_BLOCK_%d_LEVEL_%d_", ++unique_blockno, blocklevel);
        }
        PUSH(mktuple(AST_BLOCKSTART, blocklevel+1, BLOCKTYPE, strtopool(tmpname))); // enter a new block level
      } break;

      case ';': /* END */ /*  End of Block; */
      {
        //dump_comment("%c %s  blocklevel=%d for the block ending now\n", opcode, icode_name[opcode], blocklevel);
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_BLOCKEND, blocklevel)); // pass the *current* level
      } break;

      case 'Y': /* DEFAULT short */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], getshort());
        break;

      case 'A': /*  Init(Tag); */ /* INIT short */
                                  /*
         Effect:       <n> copies of the init-value are added to the list of
                       values  associated  with  the   init-variable.    The
                       init-value  is  either the default value (unassigned)
                       if the stack is empty, the  value  of  TOS  (possibly
                       converted  to  real)  if  TOS  is  a constant, or the
                       address  of  TOS  if  TOS   is   a   variable.    The
                       init-variable  is the last static object to have been
                       defined using Define.

         Error:        1. TOS, if it exists, is not of the same type as  the
                          init-variable.

         PLEASE NOTE: unlike EMAS imp, 2-D (and higher) arrays cannot be initialised!
                      (nor can records)
       */
        {
          int n_copies = getshort();
          ASTIDX init_val = -1;
          ASTIDX var;
          dump_comment("%c %s %d\n", opcode, icode_name[opcode], n_copies);

          /*
                        // N PUSHI #0  OPD_STACK = 2
                        // $ DEF V_0101 ID=LASTDEF a=17 b=1 c=33
                        // A INIT 1
0static int LASTDEF;
                */
          if (POPQ(&var)) {
            if (OP(var) == AST_ICONST || OP(var) == AST_RCONST || OP(var) == AST_ISTRINGCONST) {
              // Must be an array initialisation
              init_val = var;
              var = POP();
              if (OP(var) != AST_DECLARE) {
                dump_comment("*** UNEXPECTED TYPE IN INIT (array) ***\n");
              }
              // TO DO: Attach this init_val to var.  Will definitely need more work for array elements, but scalars should be OK.
            } else if (OP(var) == AST_DECLARE) {
              // Scalar initialisation
              if (POPQ(&init_val)) {  //init_val = POP();
                if (OP(init_val) == AST_ICONST || OP(init_val) == AST_RCONST || OP(init_val) == AST_ISTRINGCONST) {
                  //attach. but drop for now.
                } else {
                  // same as 'not found' - so initialise to unassigned pattern!  And replace the item under the stack
                  PUSH(init_val);
                }
              } else {
                // not found - so initialise to unassigned pattern!
              }
              // TO DO: Attach this init_val to var.  Will definitely need more work for array elements, but scalars should be OK.
            } else {
              dump_comment("*** UNEXPECTED TYPE IN INIT *(scalar) **\n");
            }

            PUSH(var);  // replace the AST_DECLARE but not the init value.
          } else {
            // stack is empty
            dump_comment("Did not find anything on the stack.\n");
          }

          if ((n_copies > 1) && (init_val != -1)) {
            // This is really a *very dirty* hack!  INIT n where n != 1 broke the initialisation
            // hack that relied on all the init values having been created by mktuple sequentially.
            // But it was just *so worth it* to avoid duplicating all the init values that I couldn't
            // say no to myself.  Although it was a rather heated argument and the little devil on
            // my left should did almost come to blows with the little angel on my right shoulder.
            for (int i = 1; i < n_copies; i++) {
              ASTIDX throwaway = mktuple(OP(init_val), USERFIELD(init_val, 0));  // DUP()
              COPY_FIXED_FIELDS_to_from(throwaway, init_val);
              init_val = throwaway;  // I can feel the bile rising in my throat as I write this.
              // The last item has to be the one most recently assigned to the init var
            }
            // Search ast.c for /*GASTRICREFLUX*/ to find the corresponding code.
          }

          // index into a InitValues[] array of AST indexes.
          // Let's say we have an array fred(10:12) and the
          // init values are AST tuples whose index values
          // are stored in Initvalues[34], Initvalues[35],
          // and Initvalues[36]; then INITVALUES(array)
          // would be 34 and we would assign them in the
          // declaration to fred[10], fred[11] and fred[12]
          // by extracting LB=10, UB=12 from the BOUNDS1D(array)
          // An Initvalues[n] of -1 means unassigned.
          // NOTE:
          // If I can guarantee that INIT statements are all
          // handled in order, I should be able to use the AST[] array
          // directly rather than have to duplicate the init data
          // in a separate InitValues[] array...

          // What's more, if I just assign the most recent init_val to the var
          // *and* if they are always sequential, then when printing the declaration
          // I can just work backwards from the saved index and magically find all
          // the previous inits (but in reverse order!)  So a cheap solution if it works!  Let's try it and see...

          INITVALUES(var) = init_val;
        }
        break;

      case 'p': /* ASSPAR */ /*  AssignParameter; */
      {
        // ***NOTE*** TO DO: array bound info is not yet being passed dynamically when
        // an array is passed as an arrayname parameter.  At a very minimum we must ensure
        // that it is the address of the zeroth element that is passed as the array's base
        // address so that accesses will work out correctly assuming the correct index
        // values are given within the procedure.  Other than in strings and records,
        // I don't recall that whole arrays can be passed in Imp77 by value - only by address.
        // However I do need to check that (possibly against the Vax/VMS Imp77 implementation)
        // as my memory on these things is no longer 100% trustworthy!

        // Note that arrayname parameters for both 1-D arrays and higher will require a
        // dopevector to be passed.

        ASTIDX param;
        /* Attach this parameter in TOS to the procedure descriptor in SOS */
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        param = POP();
        //debug_types(param);
        PUSH(mktuple(AST_ACTUAL_PARAMETER, param));  // leave it for the CALL to find...
      } break;

      case '@': /* PUSH tag */ /*  Stack Var(Tag); */
      {
        int tag = gettag();
        ASTIDX var = mktuple(AST_VAR, tag);
        COPY_FIXED_FIELDS_to_from(var, Descriptor[tag]);
        PUSH(var);
        dump_comment("%c %s #V_%04x  OPD_STACK = %d\n", opcode, icode_name[opcode], tag, next_opd);
      } break;

      case '^': /* PROC tag */ /*  Stack_Format = -Tag; */
      {
        int tag = gettag();
        unimplemented[opcode] = TRUE;  // warn.  I'ld like to see where this is used.
        // I don't think it *is* used, and for now I'm going to remove the AST_PROC
        // AST entry because I think it is confusing.  Easy enough to add back in
        // if it turns out to be needed after all...
        dump_comment("%c %s tag V_%04x (I THOUGHT THIS WAS AN UNUSED OPCODE!)\n", opcode, icode_name[opcode], tag);
      } break;

      case 'E': /* CALL */ /*  Call(0); */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        ASTIDX actual_parameters;
        //ASTIDX formal_parameters;
        ASTIDX tos;
        ASTIDX fn;
        int result;
        int param_count;
        ASTIDX params[32];

        param_count = 0;
        for (;;) {
          tos = POP();
          if (OP(tos) != AST_ACTUAL_PARAMETER) break;
          params[param_count++] = tos;
        }
        fn = tos;
        //debug_types(fn);

        // This is where we could force left-to-right evaluation of parameters in external calls to match
        // C/Unix calling conventions where the last parameter is pushed first.  We could assign each
        // parameter to a local temporary variable in left to right order, then call the procedure
        // using those pre-evaluated temporaries, so that it would not matter what order the parameters
        // were pushed in.

        // proc(a,b,c) -> " { typeof(a) tmp1 = a; typeof(b) tmp2 = b; typeof(c) tmp3 = c; proc(tmp1, tmp2, tmp3); } "
        
        actual_parameters = mktuple(AST_ACTUAL_PARAMETER_LIST, param_count);
        NEXT_AST += param_count;
        for (int i = 0; i < param_count; i++) {
          EXTRAFIELD(actual_parameters, (param_count - 1) - i) = params[i];  // reversed back to left-right order
        }
        EXTENDED_FIELD_COUNT(actual_parameters) = param_count;

        ACTUAL_PARAM_LIST(fn) = actual_parameters;
        //FORMAL_PARAM_LIST(fn) = formal_parameters; - should be in place already

        // fn is an AST_DECLARE tuple which if
        // passed to codegen would output the declaration.
        //debug_types(fn);

        result = mktuple(AST_CALL, fn);

        COPY_FIXED_FIELDS_to_from(result, fn);  // Only want to copy *some* fields.

        PUSH(result);
        break;
      }

      case 'G': /* ALIAS */ /*  Get String(Alias); */
      {
        // can get the same effect in C using __attribute__((alias("target"))) or __attribute__((weak, alias("target")))
        // although I'm not sure of the details or what the 'weak' is for.
        // ***NOTE*** This opcode *precedes* the '$'DEF and is optional, so there is no
        last_alias = getimpstring();  // Hacky but works
        dump_comment("%c %s %s\n", opcode, icode_name[opcode], last_alias);
      } break;

      case 'M': /* MAP */ /*  Return(Map); */
      {
        // %result == atom
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        // AST_ADDRESS_OF should return &location, or pointer, depending on the form of the parameter
        ASTIDX tag = POP();
        ASTIDX address = mktuple(AST_ADDRESS_OF, tag);
        PUSH(mktuple(AST_RESULT, address));
      } break;

      case '\'': /* Input String Value; */ /* PUSHS sconst */
      {
        char *string_value = getimpstring();  // shouldn't use C string since an IMP
            // string can potentially contain a \0 character!  But it'll do for now.
        PUSH_IMPSTRINGCONST(string_value);
        dump_comment("%c %s %s  OPD_STACK = %d\n", opcode, icode_name[opcode], string_value, next_opd);
      } break;

      case 'N': /*  Input Integer Value(0); */ /* PUSHI iconst */
      {
        int iconst = getwordconst();
        PUSH_ICONST(iconst);
        dump_comment("%c %s #%d  OPD_STACK = %d\n", opcode, icode_name[opcode], iconst, next_opd);
      } break;

      case 'D': /*  Input Real Value; */ /* PUSHR rconst */
      {
        int i, c, byte, len;

        char comment[1024];
        char realstr[1024];
        char *cp = &comment[0];
        char *rp = realstr;

        byte = get_icode(icode_file);
        assert(byte != EOF);
        len = get_icode(icode_file);
        assert(len != EOF);
        cp += sprintf(comment, "%c %s code=%d len=%d \"", opcode, icode_name[opcode], byte, len);
        c = get_icode(icode_file);
        assert(c != EOF);
        for (i = 1; i <= len; i++) {
          c = get_icode(icode_file);
          assert(c != EOF);
          if (c == '@') break;
          cp += sprintf(cp, "%c", c);
          rp += sprintf(rp, "%c", c);
        }
        if (c == '@') {
          int exponent = getshort();
          cp += sprintf(cp, " @ (signed short)0x%04x\"  OPD_STACK = %d\n", exponent, next_opd);
          rp += sprintf(cp, "@%d", exponent);
        } else {
          cp += sprintf(cp, "\"  OPD_STACK = %d\n", next_opd);
        }
        dump_comment(comment);
        PUSH_RCONST(realstr);
      } break;

      case 'O': /*  Update Line(Tag); */ /* LINE decimal */
      {
        // LINE is only called when there is nothing on the stack, so may be used
        // as a suitable place to output C code.

        // LINE will not occur in the middle of a multi-line statement, or between
        // lines which generate no ICODEs, including comments and %constinteger declarations.

        // Note that this code relies on unix file system semantics, where it is possible
        // to have two separate input streams open on the same file at the same time.
        // I don't know if Windows now supports this - in the early days when I first
        // did some C coding for Windows, only one input stream could be opened on a
        // file at a time.  So if the listing is messed up should this program ever be
        // run on Windows, that'll be why.  It can be worked around but it'll be ugly.
        FLUSH_ICODE_STACK();
        if (in_perms) {
          // This should be moved to ast.c too...
          dump_code("\n#else\n\n");
          dump_code("#include \"perms.h\"\n\n");
          dump_code("#endif // USE_PERMS_INC\n\n");
          if (PARM_OPT) {
            dump_code("#define PARM_OPT 1\n");
            dump_code("#ifdef _U // Don't do unassigned checks...\n");
            dump_code("#undef _U\n");
            dump_code("#define _U(x) (x)\n");
            dump_code("#endif\n");
          }
        }
        in_perms = FALSE;

        d = getshort();                // Target line to display up to.
        char *fname = getrawstring();  // DIRTY HACK. I redefined the 'LINE' ICODE to add
        // this extra string parameter.  But since pass1 is built-in here, it's safe.
        dump_comment("%c %s %d  OPD_STACK = %d  file=%s\n", opcode, icode_name[opcode], d, next_opd, fname);
        PUSH(mktuple(AST_IMP_LINE, d, strtopool(fname)));
      } break;

      case 'P': /* PLANT */ /*  Dump Byte(Popped Value&255); */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      case 'T': /* TRUE */  /*  Return(True); */
      case 'K': /* FALSE */ /*  Return(False); */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH_ICONST(opcode == 'T' ? 1 : 0);
        PUSH(mktuple(AST_RESULT, POP()));
      } break;

      case 'R': /* RETURN */ /*  Return(Routine); */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_RETURN));
      } break;

      case 'V': /* RESULT */ /*  Return(Fn); */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_RESULT, POP()));
      } break;

      case 'I': /*  Select Input(Icode In2);  Readsymbol(Pending); */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      case 'a': /* ACCESS */ /*  Array Access; */
        /* Add this index in TOS to the base address in SOS */
        {
          dump_comment("%c %s\n", opcode, icode_name[opcode]);
          ASTIDX tos = POP();
          ASTIDX sos = POP();
          ASTIDX array_index = mktuple(AST_ARRAYACCESS, sos, tos);
          PUSH(array_index);
          COPY_FIXED_FIELDS_to_from(array_index, sos);
          IS_ARRAY_NAME(array_index) = IS_ARRAY(array_index) = 0;
          // Indexing an array returns an array element. (Well, we'll look at 2D arrays later)
        }
        break;

      case 'n': /* SELECT tag */ /*  Select(Tag); */
      {
        int field = gettag();
        dump_comment("%c %s field #%d  OPD_STACK = %d\n", opcode, icode_name[opcode], field, next_opd);

        // currently, fields are not being displayed correctly. *** TO DO *** *** DEBUG ***

        ASTIDX OLD_DECL_AST_TAG = POP();                          // the record  *** an AST_TAG ***
        ASTIDX RECFM_AST_TAG = RECORD_FORMAT(OLD_DECL_AST_TAG);
        ASTIDX parameterlist = FORMAL_PARAM_LIST(RECFM_AST_TAG);  // TO DO: have a RECORD_FIELD_LIST instead?
        ASTIDX subfield = EXTRAFIELD(parameterlist, field - 1);
        OP(subfield) = AST_VAR;  // *VERY* HACKY!
        ASTIDX record_with_field = mktuple(AST_FIELDSELECT, OLD_DECL_AST_TAG, subfield);
        // Make 'rec_fred' have the characteristics of the 'fred' part:
        COPY_FIXED_FIELDS_to_from(record_with_field, subfield);
        PUSH(record_with_field);
      } break;

        // PLEASE NOTE!!!  'DIM' is only used for auto arrays, which cannot be initialised in Imp77.
        //                 whereas 'BOUNDS' is used for external, own, and const arrays (no matter
        //                 where they appear) (and switches) - however those must be only one-dimensional.
        //                 The 'BOUNDS' code *precedes* the $DEF but the 'DIM' code follows it.

      case 'b': /* BOUNDS */ /*  Constant Bounds; */
      {
        ASTIDX lower, upper;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        
        upper = POP();
        lower = POP();
        PUSH(mktuple(AST_BOUNDSPAIR, lower,
                     upper));  // When a $DEF of the appropriate type follows, it will pop the BOUNDS
      } break;

      case 'd': /* DIM short,short */ /*  D = Tag;   N = Tag;  Dimension(D, N); */
      {
        /*
         Instruction:  Dimension <n><d>

         Effect:       <d> pairs of integer values on the stack are used  to
                       define the bounds of the last <n> arrays to have been
                       defined.    Code   is  generated,  if  necessary,  to
                       allocate the arrays and the definitions are  adjusted
                       to reference the appropriate storage.

         Notes:        The  pairs  of  values  are  stacked  in order of the
                       declaration, that is, first dimension first.
                       In each pair of values the  lower  bound  is  stacked
                       before the upper bound.
                       The  last  <n>  tags  must have had consecutive index
                       values.

         Errors:       1. The stack contains less than 2*<d> items.
                       2. The last <n> definitions were not all arrays.

         Example:      integerarray A, B, C(1:2, Low:4)
                       Define A......
                       Define B......
                       Define C......
                       Byte 1;     Byte 2
                       Stack Low;  Byte 4
                       Dimension 3 2
       */
        int num_dimensions = getshort();
        assert(get_icode(icode_file) == ',');
        int num_decls = getshort();
        // $ DEF V_0070 id=27 a=1 b=256 c=4
        // $ DEF V_0071 id=27 a=1 b=256 c=4
        // $ DEF V_0072 id=27 a=1 b=256 c=4
        // N PUSHI #1  OPD_STACK = 4
        // N PUSHI #2  OPD_STACK = 5
        // @ PUSH #V_006f  OPD_STACK = 6
        // N PUSHI #4  OPD_STACK = 7
        // d DIM 3 2
        dump_comment("%c %s %d %d\n", opcode, icode_name[opcode], num_dimensions, num_decls);

        ASTIDX boundarray[num_dimensions];
        dump_comment("creating a %dD array dopevector\n", num_dimensions);
        for (int dim = 0; dim < num_dimensions; dim++) {
          ASTIDX high = POP();
          ASTIDX low = POP();
          boundarray[dim] = mktuple(AST_BOUNDSPAIR, low, high);
          dump_comment("Dimension %d: low %d high %d\n", dim, low, high);
        }
        int tmpdecl[num_decls];  // number of $DEF declarations on stack that these boundss should be assigned to
        for (int decls = 0; decls < num_decls; decls++) {
          ASTIDX decl = POP();
          tmpdecl[decls] = decl;

          ASTIDX dopevector = mktuple(AST_DOPEVECTOR, num_dimensions);
          NEXT_AST += num_dimensions;  // allocate extra space for bounds for each dimension
          dump_comment("assigning a dopevector %d to array declaration %d\n", dopevector, decl);
          for (int dim = 0; dim < num_dimensions; dim++) {
            EXTRAFIELD(dopevector, dim) = boundarray[dim];
          }
          DOPEVECTOR(decl) =
              dopevector;  // if non-zero, the array in the $DEF has a dopevector, otherwise use BOUNDS1D(tuple). Or add a HAS_DOPEVECTOR flag?

          // TO DO: when a multi-d array is indexed, the type info of the
          // tuple ought to be updated with a dopevector that has the appropriate
          // dimension removed.  With the current structure that won't yet work
          // because to find the dope vector we dive all the way down to the
          // underlying tag/var.  I need to propogate the dope vector and the
          // index upwards as each ARRAY_ACCESS is applied.
        }

        for (int decls = num_decls - 1; decls >= 0; decls--) {
          PUSH(tmpdecl[decls]);  // restore the declarations that were on the stack... (in the same order as they were)
        }

        break;
      }

      case 'g': /*  Test for NIL; */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      // NOTE: these might have been replaced!
      // 'q' in pass2-arm.imp handles include files:
      // c('h'):  Special Call(Tag);                                    %continue
      // c('q'):  Process Include File;                                 %continue
      case 'h': /* ALTBEG */
      case '|': /* ALT */
      case 'q': /* ALTEND */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      case 'i': /* INDEX */ /*  Array Index; */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;
        // This is actually an array indexing opcode
        // similar to 'ACCESS', for multi-dimensional arrays!
        // Some thought required as to how to handle this - do we do it
        // the C way by applying to each dimension one at a time and returning
        // an array object tuple with one less dimension encoded, *or*
        // do we save up the indexes in a list similar to the actual
        // parameters of a procedure call, and generate the C object
        // with all the indices in a single output?  Haven't yet decided.
        // *** TO DO ***
        dump_code("/* TO DO: multi-dimensional array access */");
      } break;

      case 'm': /* MONITOR */ /*  Do(Monitor Id, Monitor Ep, 0); */
      {
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        PUSH(mktuple(AST_MONITOR, 0));
      } break;

      case 'o': /*  Event Trap(Tag); */ /* ON byte short label */
      {
        short int p1 = getshort();
        assert(get_icode(icode_file) == ',');
        short int labtag;
        short int p2 = getforwardlab_from_tag(&labtag);
        dump_comment("%c %s MASK=%04x L_%04x\n", opcode, icode_name[opcode], p1, p2);
      } break;

      case 'r': /* RESOLVE m */ /*  Resolve(Tag); */
      {
        /*
r flag   String Resolution.  Flag is a three bit number, the
         bits representing:
              4    - conditional resolution.
              2    - first destination present.
              1    - second destination present.

         The resolution sets the condition code to true for
         successful resolution, and false for failure.
         The various operands are on the top of the stack.

         e.g.      S -> A.(B).C        @s@a@b@c r3

                   S -> A.(B)          @s@a@b r1

                   stop if S -> (B).C  @s@b@c r5k4s:4

                 */
        // result is tested by BF or BT!
        int flags = getshort();
        dump_comment("%c %s flags=%d\n", opcode, icode_name[opcode], flags);

        int CONDITIONAL = FALSE, OPD1 = FALSE, OPD2 = FALSE;
        ASTIDX opd1 = 0, opd2 = 0, sourcestring = 0, strtomatch = 0;
        if (flags & 1) OPD2 = TRUE;
        if (flags & 2) OPD1 = TRUE;
        if (flags & 4) CONDITIONAL = TRUE;

        if (OPD2) opd2 = POP();
        strtomatch = POP();
        if (OPD1) opd1 = POP();
        sourcestring = POP();

        ASTIDX resolve = mktuple(AST_RESOLVE, strtomatch, sourcestring, OPD1, OPD2, opd1, opd2);
        if (CONDITIONAL) {
          PUSH(mktuple(AST_CONDITIONAL_RESOLVE, resolve));
        } else {
          // TO DO: replace AST_CONDITIONAL_RESOLVE and
          // AST_UNCONDITIONAL_RESOLVE
          // with more generic calls.
          PUSH(mktuple(AST_UNCONDITIONAL_RESOLVE, resolve));
        }
      } break;

      case 's': /* STOP */ /*  To Store(Tag); */
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        //dump_code("exit(0);\n");
        PUSH(mktuple(AST_STOP));
        // a call to exit(0) would be better, but
        //  1) namespace pollution and
        //  2) not sure how to construct that call :-)
        break;

      case 'u': /* ADDA */ /*  Aop; */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      case 'c': /*  Section = 1;  Get String(Section Id); */
        unimplemented[opcode] = TRUE;
        dump_comment("%c %s\n", opcode, icode_name[opcode]);
        break;

      case 'w': /* MCODE */ /*  Machine Code;  Forget Everything; */
      {
        int i1;

        char comment[1024];
        char *cp = &comment[0];

        unimplemented[opcode] = TRUE;
        cp += sprintf(cp, "%c %s ", opcode, icode_name[opcode]);
        for (;;) {
          char *p1 = getmcstring(&i1);
          cp += sprintf(cp, " %s", p1);
          if (i1 == ';') break;
          int p2 = getshort();
          cp += sprintf(cp, " tag_%d", p2);
        }
        cp += sprintf(cp, "\n");
        dump_comment(comment);
      } break;

      // I think these just set internal flags for the compiler?  It's documented somewhere.
      case 'y': /* DIAG short */ /*  Set CD(Tag, Diag); */
      {
        int diag = getshort();
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], diag);
        break;
        PUSH(mktuple(AST_DIAGNOSE, diag));
      } break;

      case 'z': /* CONTROL short */ /*  Set CD(Tag, Control); */
      {
        int ctrl = getshort();
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], ctrl);
        break;
        PUSH(mktuple(AST_CONTROL, ctrl));
      } break;

      case 'e': /* EVENT short */ /*  Signal Event(Tag); */
      {
        int eventno = getshort();
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], eventno);
        // first value is in the tag.  optional second and third values
        // can be stacked expressions.  So we have to heuristally guess when
        // we've found the optional second or third parameters.  Fortunately (?)
        // due to the nature of the generated icode, %signal can only happen in
        // a stand-alone icode statement, so we can stop examining the stack
        // when we see the 'O' (LINE) directive.
        ASTIDX event, subevent, extra;

        event = mktuple(AST_ICONST, eventno & 15);
        if (POPQ(&subevent)) {
          if (OP(subevent) == AST_IMP_LINE) {
            PUSH(subevent);
            subevent = mktuple(AST_ICONST, 0);
            extra = mktuple(AST_ICONST, 0);
          } else {
            if (POPQ(&extra)) {
              if (OP(extra) == AST_IMP_LINE) {
                PUSH(extra);
                extra = mktuple(AST_ICONST, 0);
              } else {
                ASTIDX tmp = extra;
                extra = subevent;
                subevent = tmp;
              }
            } else
              extra = mktuple(AST_ICONST, 0);
          }
        } else {
          subevent = mktuple(AST_ICONST, 0);
          extra = mktuple(AST_ICONST, 0);
        }
        PUSH(mktuple(AST_SIGNAL, event, subevent, extra));
      } break;

      case 'l': /* LANG short */ /*  Language Flags = Tag; */
                                 // We're not going to support any language other than Imp77!
        //if ((opcode == 'l') && no_perms) suppress_perms = TRUE;  // <-- now handled in 'O' (LINE)
        dump_comment("%c %s %d\n", opcode, icode_name[opcode], getshort());
        break;

        // Unfortunately START and END can be *either*
        //  A: a record format definition *or*
        //  B: a formal parameter list

      case '{': /* START */
      {
        dump_comment("%c %s %c\n", opcode, icode_name[opcode], opcode == '{' ? '(' : ')');
        PUSH(mktuple(AST_START_PARAMLIST));

        IN_START_FINISH_GROUP = TRUE;
      } break;

      case '}': /* FINISH */
      {
        IN_START_FINISH_GROUP = FALSE;

        ASTIDX params[32];
        int param_count = 0;

        dump_comment("%c %s %c\n", opcode, icode_name[opcode], opcode == '{' ? '(' : ')');
        param_count = 0;

        for (;;) {
          ASTIDX prev = POP();
          if (OP(prev) == AST_START_PARAMLIST) break;
          params[param_count++] = prev;
        }

        // formal parameters are attached to the AST_DECLARE,
        // actual parameter to the AST_CALL ...

        ASTIDX paramlist = mktuple(AST_FORMAL_PARAMETER_LIST, param_count);
        NEXT_AST += param_count;
        for (int i = 0; i < param_count; i++) {
          OP(params[i]) = AST_DECLARE_FP;
          EXTRAFIELD(paramlist, (param_count - 1) - i) = params[i];
          // (reversed back to left-right order)
        }
        EXTENDED_FIELD_COUNT(paramlist) = param_count;

        if (last_declare_for_attaching_fields_or_parameters == -1) {
          fprintf(stderr, "START/FINISH group but no 'last_declare_for_attaching_fields_or_parameters' on record?\n");
        } else {
          // TO DO: check that last_declare_for_attaching_fields_or_parameters was a procedure or a record format...
          FORMAL_PARAM_LIST(last_declare_for_attaching_fields_or_parameters) = paramlist;
          last_declare_for_attaching_fields_or_parameters = -1;
        }
      }

      break;

      case '~':  // Used in a recordformat with a variant record, much like ALT?
                 // NOTE: This may be a newer ICODE format version!
      {
        int pending = getbyte();
        if ((' ' + 1 <= pending) && (pending <= '~')) {
          if (pending == 'A') {
            dump_comment("%c%c %s\n", opcode, pending, "ALTBEG");
            //PUSH_icode('h'); /* ALTBEG */
          } else if (pending == 'B') {
            dump_comment("%c%c %s\n", opcode, pending, "ALTEND");
            //PUSH_icode('q'); /* ALTEND */
          } else if (pending == 'C') {
            dump_comment("%c%c %s\n", opcode, pending, "ALT");
            //PUSH_icode('|'); /* ALT */
          } else {
            dump_comment("%c %s pending=%d ('%c')\n", opcode, "UNKNOWN", pending, pending);
          }
        } else {
          dump_comment("%c %s pending=%d\n", opcode, icode_name[opcode], pending);
        }
      } break; /* TODO */

        // Details of codes >= 128 can be found in
        // https://history.dcs.ed.ac.uk/archive/languages/imp77-acorn-tmp/3l/COMPILERS/bend/imp/pass2
        // Since none of them are used here they've been removed from the switch statement.

      default:
        if (opcode < 0 || opcode > 255 || icode_name[opcode] == NULL) {
          dump_comment("? OPCODE %d\n", opcode);
          break;
        } else {
          dump_comment("? %s\n", icode_name[opcode]);
          break;
        }
    }

    //if (PARM_VERBOSE) { ASTIDX tos; if (POPQ(&tos)) { debug_types(tos); PUSH(tos); } }
  }

  FLUSH_ICODE_STACK();

  // TO DO: invoke gcc!

  // cc -o zzz  -Wall -fsplit-stack -Wno-return-type -Wno-comment -g -fsanitize=undefined -fsanitize=float-divide-by-zero -fsanitize-undefined-trap-on-error -fno-sanitize-recover=all -frecord-gcc-switches -ftrapv -fsanitize=float-cast-overflow -fstack-protector -D_FORTIFY_SOURCE=1 -ggdb3  uninitcheck.c

  // gcc12 -ftrivial-auto-var-init=pattern -g -o test/paper test/paper.c perms.c

  // preferably: readelf -p .GCC.command.line zzz
  // alternatively: objdump -s -j .GCC.command.line zzz

  // running ./zzz silently invokes /snap/bin/valgrind --track-origins=yes --leak-check=full --show-leak-kinds=all --exit-on-first-error=yes -q --error-exitcode=1
  // unless the first argument is "-nv" (no valgrind).
  // Might add --merge-recursive-frames=3

  // gcc12 -ftrivial-auto-var-init=pattern -g -Wa,-adhln -O1 -o test/testfor test/testfor.c perms.c | grep -E "^(....:.*\.imp\ \*\*\*\*|...[0-9]......[0-9A-F][0-9A-F])" test/testfor.asm|ecce - - -command "f/1:/l0k-0;m0f-/%end/mk0;%c"

  int version = -1;
  char gcc[1024] = {0};
  char *GCC = "cc";

  sysnprintf(gcc, 1024, "gcc -dumpversion");
  if (*gcc != '\0') {
    char *s = strchr(gcc, '.');
    if (s) *s = '\0';
    version = atoi(gcc);
    GCC = "gcc";
  }

  if (version < 0) {
    sysnprintf(gcc, 1024, "cc -dumpversion");
    if (*gcc != '\0') {
      char *s = strchr(gcc, '.');
      if (s) *s = '\0';
      version = atoi(gcc);
      GCC = "cc";
    }
  }

  if (version < 12) {
    // this one's just for me and my non-standard installation
    sysnprintf(gcc, 1024, "gcc12 -dumpversion");
    if (*gcc != '\0') {
      char *s = strchr(gcc, '.');
      if (s) *s = '\0';
      version = atoi(gcc);
      GCC = "gcc12";
    }
  }

  int old_gcc = (version < 12);
  if (object_filename[0] == '\0') {
    if (PARM_LINK) {
      sprintf(object_filename, "%s", base_filename);
    } else {
      sprintf(object_filename, "%s.o", base_filename);
    }
  }

  // Once the translator is working better, we can invoke the C compiler
  // from i2c, using system() or popen().

  // i2c -O and cc -DPARM_OPT -O2 with no -g and running with --nv is the fastest.
  // i2c and cc -ftrivial-auto-var-init=pattern -g is best for catching errors

  // extra checking can be asked for by adding these to the gcc command:
  //  -Wall 
  //  -frecord-gcc-switches 

  //  -fsanitize=undefined 
  //  -fsanitize=float-divide-by-zero 
  //  -fsanitize-undefined-trap-on-error 
  //  -fsanitize=float-cast-overflow 
  //  -fno-sanitize-recover=all
  //  -fsplit-stack 
  //  -fstack-protector 
  //  -Wno-return-type 
  //  -Wno-comment 
  //  -ftrapv
  //  -D_FORTIFY_SOURCE=[123]         1 - compile time   2 - run time  3 - extra features is using gcc v12
  //  -ggdb3                          extra debugging for gdb including macro definitions
  
  if (PARM_OPT || old_gcc || strcmp(output_filename, "/dev/stdout") == 0) {
    if (PARM_LINK) {
      fprintf(stderr, "cc -g -DPARM_OPT -fmerge-constants -o %s %s.c perms.c\n", object_filename, base_filename);
    } else {
      fprintf(stderr, "cc -g -DPARM_OPT -fmerge-constants -o %s -c %s.c\n", object_filename, base_filename);
    }
    fprintf(stderr, "( you might prefer to compile with %s --check --gdb %s.imp )\n", PROGNAME, base_filename);
  } else {
    if (PARM_LINK) {
      fprintf(stderr, "%s -ftrivial-auto-var-init=pattern -g -o %s %s perms.c\n", GCC, object_filename,
              output_filename);
    } else {
      fprintf(stderr, "%s -ftrivial-auto-var-init=pattern -g -c -o %s %s\n", GCC, object_filename,
              output_filename);
    }
  }

  {
    int i, any = 0;
    for (i = 0; i < 128; i++)
      if (unimplemented[i]) any = 1;
    if (any) {
      fprintf(stderr, "Remaining unimplemented icodes needed to compile this program:\n  ");
      for (i = 0; i < 128; i++) {
        if (unimplemented[i]) fprintf(stderr, " %s", icode_name[i]);
      }
      fprintf(stderr, "\n");
    }
  }

  exit(0);
  (void)NULL_TUPLE;
  (void)NULL_LABEL;
  return (0);
}
