#define IN_PERMS_C 1
#include "perms.h"

#include <limits.h> // PATH_MAX for valgrind code
#include <time.h> // strftime
int     _imp_INSTREAM = 0,  _imp_OUTSTREAM = 0;
FILE    *_imp_INFILE,       *_imp_OUTFILE;

_imp_filedata _imp_infile[256];
_imp_filedata _imp_outfile[256];

_imp_string   _imp_promptstr;

// similar to C's __LINE__ and __FILE__ for diagnostics:
int _imp_current_line;
char *_imp_current_file;

const double PI = 3.141592653589793238462;
const _imp_string1 SNL = {.charno = {1, '\n'} };

static char *PROGNAME = "<imp main program>";
  
eventfm EVENT; // global
_imp_on_event_handler *global_handler;

static void _imp_init_files(void) {
  int i;
  for (i = 0; i < 256; i++) {
    _imp_infile[i].f = NULL;
    _imp_infile[i].fname = _imp_str_literal("<null>");
    _imp_infile[i].streamno = i;
    _imp_infile[i].lastchar = '\n';
    _imp_infile[i].nextchar = -1;
    _imp_infile[i].inpos = 0;
    _imp_outfile[i].f = NULL;
    _imp_outfile[i].fname = _imp_str_literal("<null>");
    _imp_outfile[i].streamno = i;
    _imp_outfile[i].lastchar = '\n';
    _imp_outfile[i].nextchar = -1;
    _imp_outfile[i].outpos = 0;
  }
  _imp_infile[0].f = fopen("/dev/tty", "r");
  _imp_infile[0].fname = _imp_str_literal("<console>");

  _imp_outfile[0].f = fopen("/dev/tty", "w"); // or could use stderr
  _imp_outfile[0].fname = _imp_str_literal("<console>");

  _imp_infile[0].inpos = 0;

  _imp_infile[1].f = stdin;
  _imp_infile[1].fname = _imp_str_literal("<stdin>");

  _imp_outfile[1].f = stdout;
  _imp_outfile[1].fname = _imp_str_literal("<stdout>");

  _imp_outfile[1].outpos = 0;

  _imp_INSTREAM = 1; _imp_OUTSTREAM = 1;
  _imp_INFILE  = _imp_infile[_imp_INSTREAM].f;
  _imp_OUTFILE = _imp_outfile[_imp_OUTSTREAM].f;
}


// A system() analog that behaves like sprintf, to make it easier
// to access the output of a system command from C.
// This is an initial simple version, but ideally it should be
// expanded to allow stdargs formatting of the command string.
int sysnprintf(char *out, int maxlen, char *command) {
  FILE *pipe;
  pipe = popen(command, "r");
  for (int i = 0; i < maxlen-1; i++) {
    int c = fgetc(pipe);
    if (c == EOF) { out[i] = '\0'; pclose(pipe); return i; }
    out[i] = c;
  }
  return 0;
}


#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <valgrind/valgrind.h>   /* for RUNNING_ON_VALGRIND */

/*
 * Put this at the very start of main(), before you do anything else.
 *
 * int main(int argc, char **argv) {
 *     restart_under_valgrind_if_needed(argc, argv);
 *     ...
 * }
 */

void restart_under_valgrind_if_needed(int argc, char **argv)
{
#ifdef PARM_OPT
    return;  // Only invoke valgrind if checks are requested
#endif
  
    // even if compiled for valgrind support, don't invoke it if the first
    // command-line option was --nv:
    
    if ((argc > 1) && (strcmp(argv[1], "--nv")==0)) {
      for (int i = 1; i <= argc; i++) {
        argv[i] = argv[i+1]; // will include terminal NULL of argv[argc]
      }
      return;
    }
    
    if (RUNNING_ON_VALGRIND) {            /* already under Valgrind? */
      return;
    }

    /* Prevent infinite recursion if valgrind itself re-execs, etc. */
    if (getenv("RUN_UNDER_VALGRIND_DONE")) {
      return;
    }
    setenv("RUN_UNDER_VALGRIND_DONE", "1", 1);

    /* Build new argv: [valgrind, options..., original_argv[0..]] */

    // TO DO:  As well as checking that valgrind is available, we need to
    // check that the version is >= valgrind-3.14 which is the first version
    // that supports the --exit-on-first-error= option, necessary to make the
    // output look like what Imp users expect.

    const char *vg;
    char version[1024] = { 0 };
    sysnprintf(version, 1024, "valgrind --version");
    if ((strncmp(version, "valgrind", strlen("valgrind")) != 0) || (strcmp(version, "valgrind-3.13") > 0)) {
      version[0] = '\0';
      sysnprintf(version, 1024, "/snap/bin/valgrind --version");
      if (strncmp(version, "valgrind", strlen("valgrind")) == 0) {
        vg = "/snap/bin/valgrind";
      } else {
        fprintf(stderr, "? Warning: valgrind not available.  Running without checks.\n");
        return;
      }
    } else vg = "valgrind";
    
    /* Fixed options you asked for. */
    const char *vg_args_fixed[] = {
        "--track-origins=yes",
        "--leak-check=full",
        // "--show-leak-kinds=all", // changed to tidy up some of the valgrind messages...
        "--show-leak-kinds=none",
        "--exit-on-first-error=yes",
        "-q",
        "--error-exitcode=1",
        NULL
    };

    /* Count fixed options. */
    int vg_fixed_count = 0;
    while (vg_args_fixed[vg_fixed_count] != NULL) {
        vg_fixed_count++;
    }

    /* New argv size: valgrind + fixed + original argc + NULL */
    int new_argc = 1 + vg_fixed_count + argc;
    char **new_argv = calloc((size_t)new_argc + 1, sizeof(char *));
    if (!new_argv) {
        perror("calloc");
        exit(1);
    }

    int idx = 0;
    new_argv[idx++] = (char *)vg;            /* argv[0] = valgrind path */

    for (int i = 0; i < vg_fixed_count; i++) {
        new_argv[idx++] = (char *)vg_args_fixed[i];
    }

    /* Append original argv[0..argc-1] */
    for (int i = 0; i < argc; i++) {
        new_argv[idx++] = argv[i];
    }

    new_argv[idx] = NULL;

    /* Replace current process image with valgrind + our program. */
    if (*vg == '/' || *vg == '.') {
      execv(vg, new_argv);
    } else {
      execvp(vg, new_argv);
    }

    /* If execv returns, it's an error. */
    fprintf(stderr, "%s: %s - %s\n", PROGNAME, strerror(errno), vg);
    //perror("execv valgrind");
    exit(1);
}

#include <execinfo.h>  // backtrace, backtrace_symbols_fd
#include <valgrind/valgrind.h>  // Just for RUNNING_ON_VALGRIND
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>

void _imp_generic_backtrace(void) {
  if (!RUNNING_ON_VALGRIND) {
    fprintf(stderr, "BACKTRACE NOT AVAILABLE\n\n");
    exit(1);
  }

  // Get exe path
  char exe_path[PATH_MAX];
  ssize_t len = readlink("/proc/self/exe", exe_path, sizeof(exe_path) - 1);
  if (len == -1) {
    fprintf(stderr, "backtrace: readlink failed\n\n");
    exit(1);
  }
  exe_path[len] = '\0';

  // Capture backtrace
  void *trace[32];
  int n_ips = backtrace(trace, 32);
  if (n_ips <= 1) {  // Skip this frame
    fprintf(stderr, "backtrace failed\n\n");
    exit(1);
  }

  // Declare linker symbols (GNU ld provides these)
  extern char __executable_start[], etext[];
  
  for (int i = 1; i < n_ips; i++) {  // Skip IMP_SIGNAL frame
    
    uintptr_t ip = (uintptr_t)trace[i];
    if (ip < (uintptr_t)__executable_start || ip > (uintptr_t)etext) continue;
    
    //uintptr_t ip = (uintptr_t)trace[i];
    //if (ip < 0x8048000 || ip > 0x8050000) continue;  // App range (adjust)
    
    char cmd[512];
    snprintf(cmd, sizeof(cmd), "addr2line -e %s -iC -f -p %p", exe_path, trace[i]);

    FILE *pipe = popen(cmd, "r");
    if (pipe) {
      char linebuf[512];
      if (fgets(linebuf, sizeof(linebuf), pipe)) {
        linebuf[strcspn(linebuf, "\n")] = 0;
        if (strncmp(linebuf, "_signal_event ", strlen("_signal_event ")) == 0) {
        } else if (strncmp(linebuf, "_start ", strlen("_start ")) == 0) {
        } else if (strncmp(linebuf, "_imp_", strlen("_imp_")) == 0) {
        } else {
          fprintf(stderr, "    Entered from %s\n", linebuf);
        }
      }
      pclose(pipe);
    }
  }
}

void _imp_fallback_signal_handler(int eventno, int subevent, int extra, char *message) {
  fprintf(stderr, "\n* Monitor entered from IMP - %%signal %d,%d,%d  %s\n", eventno, subevent, extra, message);
  _imp_generic_backtrace();
  exit(1);
}

// trace entry and exit from procedures as a debugging aid.  Not used much.
// Backtrace could use this info too if saved on a stack but better to use
// proper system backtrace if available.  Still to do: add a command-line
// option to invoke runtime tracing.

static int _imp_trace_depth = 0;
int _imp_trace_enter(int line, char *file, char *funcname) {
  for (int i = _imp_trace_depth; i > 0; i--) fputc(' ', stderr);
  fprintf(stderr, "\"%s\", %d: > %s\n", file, line, funcname);
  _imp_trace_depth += 1;
}

int _imp_trace_exit(int line, char *file, char *funcname) {
  _imp_trace_depth -= 1;
  for (int i = _imp_trace_depth; i > 0; i--) fputc(' ', stderr);
  fprintf(stderr, "\"%s\", %d: < %s\n", file, line, funcname);
}

void _imp_monitor(int n, int line, char *file, const char *funcname) {
  if (n <= 0) {
    char *fun;
    if (file == NULL) file = "<unknown file>";
    if (funcname == NULL) fun = "<unknown function>"; else fun = (char *)funcname;
    fprintf(stderr, "\n* %%MONITOR entered from IMP in %s at %s:%d\n", *funcname == '\0' ? "block" : fun, file, line);
  } else {
    fprintf(stderr, "\n* %%MONITOR %d entered from IMP in %s at %s:%d\n", n, *funcname == '\0' ? "block" : funcname, file, line);
  }
 _imp_generic_backtrace();
}

void _imp_initialise(int argc, char **argv) {
  PROGNAME = argv[0];
  restart_under_valgrind_if_needed(argc, argv);
  _imp_init_files(); // Initialise Imp I/O streams.
}

void _imp_issue_prompt(void) {
  if (_imp_OUTSTREAM == 0 && _imp_outfile[0].lastchar == '\n' && _imp_INSTREAM == 0 && _imp_infile[0].lastchar == '\n') {
    PRINTSTRING(_imp_promptstr); fflush(_imp_OUTFILE);
    if (*LENGTH(&_imp_promptstr) > 0) _imp_outfile[_imp_OUTSTREAM].lastchar = *CHARNO(&_imp_promptstr, *LENGTH(&_imp_promptstr)-1);
  }
}

int _imp_resolve(_imp_string s, _imp_string *left, _imp_string match, _imp_string *right);
_imp_string _imp_join(_imp_string left, _imp_string right);
int _imp_strcmp(_imp_string left, _imp_string right);

eventfm EVENT; // global

_imp_on_event_handler *global_handler; // only declared and installed in an onevent block!

void _signal_event(char *file, int line, _imp_string info, int event, int subevent, int extra) {
  _imp_fallback_signal_handler(event, subevent,  extra, "");
}

int _imp_caught_on_event(int event, int bitpos, ...) {
  // resignals if not in catch-list.
}

//int           eventmask(int bitpos, ...);

int _imp_caught_fault(int event, int bitpos, ...) {
  // does not signal. just returns true/false
}

//void _imp_fault(const char *format, ...) {
//  // calls generated by the compiler. Not for users.
//}

//void _imp_warn(const char *format, ...) {
//}

void _imp_readbyte(unsigned char *dest) {
  int num;
  int rc = fscanf(_imp_INFILE, "%d", &num);
  // if (rc != 1) _imp_signal(?,?,?);
  // first do a range check for 0:255 (or possibly -128:127 or even -128:255)
  *dest = num;
}

void _imp_readshort(short int *dest) {
  int num;
  int rc = fscanf(_imp_INFILE, "%hd", dest);
  // if (rc != 1) _imp_signal(?,?,?);
  // first do a range check for 0x8000 to 0x7FFF
  *dest = num;
}

void _imp_readint(int *dest) {
  int num;
  int rc = fscanf(_imp_INFILE, "%d", dest);
  // if (rc != 1) _imp_signal(?,?,?);
  // first do a range check for 0x80000000 to 0x7FFFFFFF
  *dest = num;
}

void _imp_readlong(long long int *dest) {
  int num;
  int rc = fscanf(_imp_INFILE, "%lld", dest);
  // if (rc != 1) _imp_signal(?,?,?);
  *dest = num;
}

void _imp_readfloat(float *dest) {
  double num;
  int rc = fscanf(_imp_INFILE, "%f", dest);
  // if (rc != 1) _imp_signal(?,?,?);
  // possible warning for loss of precision?
  *dest = num;
}

void _imp_readdouble(double *dest) {
  double num;
  int rc = fscanf(_imp_INFILE, "%lf", dest);
  // if (rc != 1) _imp_signal(?,?,?);
  *dest = num;
}

void _imp_readsymbol(int *P) { *P = fgetc(_imp_INFILE); }
void _imp_readch(int *P) { *P = fgetc(_imp_INFILE); }
int _imp_nextsymbol(void) { int c = fgetc(_imp_INFILE); ungetc(c, _imp_INFILE); return c; }
int _imp_nextch(void) { int c = fgetc(_imp_INFILE); ungetc(c, _imp_INFILE); return c; }
void _imp_skipsymbol(void) { (void) fgetc(_imp_INFILE); }
void _imp_printsymbol(char SYM) { fputc(SYM, _imp_OUTFILE); }
void _imp_printch(char SYM) { fputc(SYM, _imp_OUTFILE); }
void _imp_printstring(_imp_string S) { fprintf(_imp_OUTFILE, "%*s", S.length, S.cstr.s); }
void _imp_write(int V, int P) { fprintf(_imp_OUTFILE, "%*d", P, V); } // or swap
void _imp_readitem(_imp_string *S) { }
void _imp_readstring(_imp_string *S) { }
void _imp_readtext(_imp_string *S, int DELIM) { }
_imp_string _imp_nextitem(void) { }
void _imp_readline(_imp_string *S) { }
int _imp_instream(void) { return _imp_INSTREAM; }
int _imp_outstream(void) { return _imp_OUTSTREAM; }
int _imp_inputstream(void) { return _imp_INSTREAM; }
int _imp_outputstream(void) { return _imp_OUTSTREAM; }
_imp_string _imp_inputname(void) { _imp_infile[_imp_INSTREAM].fname; }
_imp_string _imp_outputname(void) { _imp_outfile[_imp_OUTSTREAM].fname; }
_imp_string _imp_infilename(void) { _imp_infile[_imp_INSTREAM].fname; }
_imp_string _imp_outfilename(void) { _imp_outfile[_imp_OUTSTREAM].fname; }
void _imp_selectinput(int N) { }
void _imp_selectoutput(int N) { }

void _imp_openinput(int N, _imp_string FD) {
  FILE *f;
  // Should we warn if there is already a stream open on this channel? (probably)
  f = fopen(_imp_i2cstr(&FD), "r");
  if (f == NULL) {
    //_imp_signal(9,0,N,strerror(errno)); // if signals not enabled, drop through
  }
  _imp_infile[N].f = f;
  _imp_infile[N].fname = FD;
  _imp_infile[N].streamno = N;
  _imp_infile[N].lastchar = '\n';
  _imp_infile[N].nextchar = -1;
  _imp_infile[N].inpos = 0;
}

void _imp_openbinaryinput(int N, _imp_string FD) {
  FILE *f;
  f = fopen(_imp_i2cstr(&FD), "rb");
  if (f == NULL) {
    //_imp_signal(9,0,N,strerror(errno)); // if signals not enabled, drop through
  }
  _imp_infile[N].f = f;
  _imp_infile[N].fname = FD;
  _imp_infile[N].streamno = N;
  _imp_infile[N].lastchar = '\n';
  _imp_infile[N].nextchar = -1;
  _imp_infile[N].inpos = 0;
}

void _imp_openoutput(int N, _imp_string FD) {
  FILE *f;
  f = fopen(_imp_i2cstr(&FD), "w");
  if (f == NULL) {
    //_imp_signal(9,0,N,strerror(errno)); // if signals not enabled, drop through
  }
  _imp_outfile[N].f = f;
  _imp_outfile[N].fname = FD;
  _imp_outfile[N].streamno = N;
  _imp_outfile[N].lastchar = '\n';
  _imp_outfile[N].outpos = 0;
}

void _imp_openbinaryoutput(int N, _imp_string FD) {
  FILE *f;
  f = fopen(_imp_i2cstr(&FD), "wb");
  if (f == NULL) {
    //_imp_signal(9,0,N,strerror(errno)); // if signals not enabled, drop through
  }
  _imp_outfile[N].f = f;
  _imp_outfile[N].fname = FD;
  _imp_outfile[N].streamno = N;
  _imp_outfile[N].lastchar = '\n';
  _imp_outfile[N].outpos = 0;
}

void _imp_defineinput(int I, _imp_string SPEC) { }
void _imp_defineoutput(int I, _imp_string SPEC) { }

void _imp_closeinput(void) {
  int rc;
  // TO DO: check validity of _imp_INSTREAM
  rc = fclose(_imp_infile[_imp_INSTREAM].f); // aka INFILE
  _imp_infile[_imp_INSTREAM].f = _imp_INFILE = NULL;
  if (rc != 0) _imp_signal(9, 0, _imp_INSTREAM, strerror(errno));
  // if signals not enabled, drop through
  _imp_INSTREAM = -1;
}

void _imp_closeoutput(void) {
  int rc;
  // TO DO: check validity of _imp_OUTSTREAM
  rc = fclose(_imp_outfile[_imp_OUTSTREAM].f); // aka OUTFILE
  _imp_outfile[_imp_OUTSTREAM].f = _imp_OUTFILE = NULL;
  if (rc != 0) _imp_signal(9, 1, _imp_OUTSTREAM, strerror(errno));
  // if signals not enabled, drop through...
  _imp_OUTSTREAM = -1;
  // or should we default to stream 0 on closing any other stream?
  // While disallowing closing of stream 0?
  // also do OUTFILE = NULL; or OUTFILE = stderr; ?
}

void _imp_abandoninput(void) { } // ABANDON INPUT and OUTPUT not yet added to perms.h
void _imp_abandonoutput(void) { }

void _imp_resetinput(void) {
  fseek(_imp_infile[_imp_INSTREAM].f, 0L, SEEK_SET);
}
// *Is* there a RESET OUTPUT?

void _imp_inputposition(void) {    // ftell
  fprintf(stderr, "*** ERROR: Empty version of %s called!\n", __PRETTY_FUNCTION__);
}

void _imp_outputposition(void) {
  fprintf(stderr, "*** ERROR: Empty version of %s called!\n", __PRETTY_FUNCTION__);
}

void _imp_setinput(int P) {
  fseek (_imp_INFILE, P, SEEK_SET);
}
void _imp_setoutput(int P) {
  fseek (_imp_OUTFILE, P, SEEK_SET);
}

void _imp_positioninput(int P) {     // fseek
  _imp_setinput(P);
}
void _imp_positionoutput(int P) {
  _imp_setoutput(P);
}


void _imp_space(void) { fprintf(_imp_OUTFILE, " "); }
void _imp_spaces(int N) { while (N --> 0) fprintf(_imp_OUTFILE, " "); }
void _imp_newpage(void) { fprintf(_imp_OUTFILE, "\f"); }
void _imp_newline(void) { fprintf(_imp_OUTFILE, "\n"); }
void _imp_newlines(int N) { while (N --> 0) fprintf(_imp_OUTFILE, "\n"); }
void _imp_print(double R, int BEFORE, int AFTER) { fprintf(_imp_OUTFILE, "%*f", AFTER, R); }
void _imp_printfloating(double R, int BEFORE, int AFTER) { fprintf(_imp_OUTFILE, "%*f", AFTER, R); } // TO DO
void _imp_printfl(double R, int PLACES) { fprintf(_imp_OUTFILE, "%*f", PLACES, R); }
_imp_string _imp_substring(_imp_string S, int FROM, int TO) {
    int GET;
    int PUT;
    _imp_string TEMP;
    if (FROM < 0) goto L_0002;
    if (FROM <= *LENGTH(&S)) goto L_0003;
  L_0002:
    _imp_signal(6, 2, FROM, "");
  L_0003:
    if (TO < 0) goto L_0004;
    if (TO <= *LENGTH(&S)) goto L_0005;
  L_0004:
    _imp_signal(6, 2, TO, "");
  L_0005:
    if (FROM <= TO) goto L_0006;
    _imp_signal(5, 3, 0, "");
  L_0006:
    *LENGTH(&TEMP) = (TO - FROM) + 1;
    PUT = 1;
    GET = FROM;
  L_0007:
    if (GET > TO) goto L_0008;
    *CHARNO(&TEMP, PUT) = *CHARNO(&S, GET);
    PUT = PUT + 1;
    GET = GET + 1;
    goto L_0007;
  L_0008:
    return TEMP;
    ;
}
_imp_string _imp_fromstring(_imp_string S, int FROM, int TO) { return _imp_substring(S, FROM, TO); }
_imp_string _imp_trim(_imp_string S, int MAX) {
  if (MAX >= 0) goto L_0002;
  _imp_signal(6, 2, MAX, "");
  L_0002:
  if ( /*%map*/ * LENGTH( &S) <= MAX) goto L_0003;
  /*%map*/ * LENGTH( &S) = MAX;
L_0003:
  return S;
}

_imp_string _imp_time(void) {
  char temp[256];
  time_t rawtime;
  struct tm *ptm;
  
  if ((rawtime=time(NULL)) == -1)        { _imp_signal(0,0,0, "time() failed"); exit(EXIT_FAILURE); }
  if ((ptm=localtime(&rawtime)) == NULL) { _imp_signal(0,0,0, "localtime() failed"); exit(EXIT_FAILURE); }
  strftime(temp, 255, "%T", ptm);
  return _imp_c2istr(temp);
}

_imp_string _imp_date(void) {
  char temp[256];
  time_t rawtime;
  struct tm *ptm;

  if ((rawtime=time(NULL)) == -1)        { _imp_signal(0,0,0, "time() failed"); exit(EXIT_FAILURE); }
  if ((ptm=localtime(&rawtime)) == NULL) { _imp_signal(0,0,0, "localtime() failed"); exit(EXIT_FAILURE); }
  strftime(temp, 255, "%D", ptm);
  return _imp_c2istr(temp);
}

int _imp_cputime(void) {
  return clock();
}

_imp_string _imp_cliparam(void) {
#ifdef IMP_SOURCE
  %external %string (255) %fn  CLI PARAM %alias "_imp_cliparam"
    %string (255) result = ""
    %integer p
    %for p = 1, 1, getargcount-1 %cycle
      %if length(result)+length(getarg(p)) > 255 %start
        %result = result.substring(getarg(p),1,255-length(result))
        ! Silently truncate, or %signal an overflow event?
      %finish
      result <- result.getarg(p)
      result <- result." " %if p < getargcount-1 %and length(result) <= 254
    %repeat
    %result = result
  %end
                                                          
#endif
}

// Awkward name clash between different Imp versions. In one, EVENT is a record;
// in another, EVENT is an integerfn returning what the other calls EVENT_EVENT
// We might be able to hackily support both by using a macro for the function style
// one?

int _imp_subevent(void) {
  return EVENT.SUBEVENT;
}

int _imp_eventinfo(void) {
  return EVENT.EXTRA;
}

_imp_string _imp_eventmessage(void) { return EVENT.MESSAGE; }

_imp_string _imp_itos(int V, int P) {
  //char temps[256];
  //sprintf(temps, "%*d", Places, I);
  // +1? I have a program somewhere that validates against various original Imp versions.
  //return _imp_c2istr(temps);

int VV;
int Q;
int POS;
                                                                //      6    %byteintegerarray store(0:127)
unsigned char STORE[(127)-(0)+1];
                                                                //      7    
                                                                //      8    vv = v;  vv = -vv %if vv > 0
VV = V;
if (VV <= 0) goto L_0002;
VV = (-(VV));
L_0002:
                                                                //      9    pos = 127
POS = 127;
                                                                //     10    %while vv <= -10 %cycle
L_0003:
if (VV > (-(10))) goto L_0004;
                                                                //     11      q = vv//10
Q = (int)(VV) / (int)(10);
                                                                //     12      store(pos) = q*10-vv+'0';  pos = pos-1
STORE[POS] = (Q * 10) - VV + 48;
POS = POS - 1;
                                                                //     13      vv = q
VV = Q;
                                                                //     14    %repeat
goto L_0003;
L_0004:
                                                                //     15    store(pos) = '0'-vv
STORE[POS] = 48 - VV;
                                                                //     16    %if p <= 0 %start
if (P > 0) goto L_0006;
                                                                //     17      p = 128+p
P = 128 + P;
                                                                //     18    %else
goto L_0007;
L_0006:
                                                                //     19      p = 128-p
P = 128 - P;
                                                                //     20      p = pos %if p > pos
if (P <= POS) goto L_0008;
P = POS;
L_0008:
                                                                //     21      p = p-1
P = P - 1;
                                                                //     22    %finish
L_0007:
                                                                //     23    pos = pos-1 %and store(pos) = '-' %if v < 0
if (V >= 0) goto L_0009;
POS = POS - 1;
STORE[POS] = 45;
L_0009:
                                                                //     24    %while pos > p %and pos > 1 %cycle
L_000a:
if (POS <= P) goto L_000b;
if (POS <= 1) goto L_000b;
                                                                //     25      pos = pos-1;  store(pos) = ' '
POS = POS - 1;
STORE[POS] = 32;
                                                                //     26    %repeat
goto L_000a;
L_000b:
                                                                //     27    pos = pos-1;  store(pos) = 127-pos
POS = POS - 1;
STORE[POS] = 127 - POS;
                                                                //     28    %result = string(addr(store(pos)))
return  /*%map*/ * STRING(ADDR( &STORE[POS]));
}

#ifndef PARM_OPT
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> // for readlink

static void __attribute__ ((noinline)) _imp_report_unass_error(char *vs) {

  // This could eventually be moved to the default signal handler, but for
  // now we'll skip raising a signal and report it directly.  No use yet of
  // libbacktrace, but just reporting the line and file is going to be
  // pretty helpful.
  
  void *caller_addr = __builtin_return_address(1);
  char exe_path[1024];
  char addr_str[32];
  FILE *pipe;
  char func_line[512] = { '\0' };
  char loc_line[256];
  ssize_t len;
  char cmd[2048];

  // or could use global imp_argv[0] below:
  if ((len = readlink("/proc/self/exe", exe_path, sizeof(exe_path)-1)) == -1) { perror("readlink"); exit(1); }
  exe_path[len] = '\0';

  sprintf(addr_str, "0x%lx", (unsigned long)caller_addr);

  snprintf(cmd, sizeof(cmd), "addr2line -e %s -f %s", exe_path, addr_str);
  if (!(pipe = popen(cmd, "r"))) { perror("popen"); exit(1); }

  char *rc=fgets(func_line, sizeof(func_line), pipe);(void)rc;
  if (fgets(loc_line, sizeof(loc_line), pipe)) {
    loc_line[strcspn(loc_line, "\n")] = 0;
    fprintf(stderr, "\n* Monitor entered from IMP - %s has never been assigned a value\n", vs);
  }
  pclose(pipe);

  _imp_generic_backtrace();
  
  exit(1);
}

void _imp_report_range_error(int ix, int low, int high, char *arrayname) {

  // This could eventually be moved to the default signal handler, but for
  // now we'll skip raising a signal and report it directly.  No use yet of
  // libbacktrace, but just reporting the line and file is going to be
  // pretty helpful.
  
  void *caller_addr = __builtin_return_address(1);
  char exe_path[1024];
  char addr_str[32];
  FILE *pipe;
  char func_line[512] = { '\0' };
  char loc_line[256];
  ssize_t len;
  char cmd[2048];

  // or could use global imp_argv[0] below:
  if ((len = readlink("/proc/self/exe", exe_path, sizeof(exe_path)-1)) == -1) { perror("readlink"); exit(1); }
  exe_path[len] = '\0';

  sprintf(addr_str, "0x%lx", (unsigned long)caller_addr);

  snprintf(cmd, sizeof(cmd), "addr2line -e %s -f %s", exe_path, addr_str);
  if (!(pipe = popen(cmd, "r"))) { perror("popen"); exit(1); }

  char *rc=fgets(func_line, sizeof(func_line), pipe);(void)rc;
  if (fgets(loc_line, sizeof(loc_line), pipe)) {
    loc_line[strcspn(loc_line, "\n")] = 0;
    fprintf(stderr, "\n* Monitor entered from IMP - index %d out of range for array %s(%d:%d)\n", ix, arrayname, low, high);
  }
  pclose(pipe);

  _imp_generic_backtrace();
  
  exit(1);  
}

// Calls to these procedures are inserted in the user program so the call itself should be kept short!
// I.e. minimise unnecessary extra parameters and runtime overhead.  Can be made inline for a marginal
// speed gain at the expense of a marginal size loss.  If inlined, these have to be in perms.h not perms.c
int __attribute__ ((noinline)) _imp_rcheck1d(int ix, int low, int high, char *arrayname) {
  if (ix < low || ix > high) _imp_report_range_error(ix, low, high, arrayname);
  return ix;
}

int __attribute__ ((noinline)) _imp_ucheck_int(int v, char *vs) {
  int unass; // trying to be a little bit portable rather than hard-coding 0xfefefefe
             // but this does risk checking against some random value (which might turn
             // out to be '0' if not compiled with -ftrivial-auto-var-init=pattern )
  if (v == unass) _imp_report_unass_error(vs);
  return v;
}

// because this is inserted into machine generated code,
// we know that 'v' will always be an atom, not an expression:
#define _R(ix, low, high, vs) _imp_rcheck1d(ix, low, high, vs)
#define _U(v) _imp_ucheck_int(v, #v)
#define _US(v,S) _imp_ucheck_int(v, S)

#else
// Not needed as this is also done in perms.h:
//#warning "This code has been compiled without runtime checks.  Remove -DPARM_OPT to enable them."
#define _R(ix, low, high, vs) (ix)
#define _U(v) (v)
#define _US(v,S) (v)

#endif
