/*
 *      ISERVER  -  INMOS standard file server
 *
 *      acsilink.c
 *
 *      Link module for Atari ST with Atari's ACSI link adaptor board
 *
 *      Copyright (c) INMOS Ltd., 1991.
 *      All Rights Reserved.
 */

/* Modification:
 * 08/03/91 BJ  Initial coding
 */

#include <stdio.h>
#include <osbind.h>

#include "inmos.h"
#include "iserver.h"

#include "traps.h"

/* #define TRACE */

/* Bits for _LINKMODE and _LNKSWITCH: */
#define ROOTLINK     0     /* Connect link adaptor to root transputer */
#define CONFIGLINK   1     /* Connect link adaptor to configuration link */

#define FAST         0     /* Run link at 20Mb/s */
#define SLOW         2     /* Run link at 10Mb/s */

#define YES          1
#define NO           0

#define TRAP8VEC     (0x20 + 8)

#define NULL_LINK          -1

static int SetTimeout (int t);
static int Installed (void);

static LINK ActiveLink = NULL_LINK;


/* OpenLink --- open a link to the transputer */

LINK OpenLink (Name)
char *Name;
{
   if (ActiveLink != NULL_LINK)
      return (ER_LINK_CANT);     /* Already open */
            
   if (Installed () == NO)       /* No resident driver s/w */
      return (ER_LINK_CANT);
      
   if ((Name == NULL) || (*Name == '\0'))
      trap8 (_LNKSWTCH, ROOTLINK, 0);       /* Default speed */
   else {
      if ((*Name == 'f') || (*Name == 'F'))
         trap8 (_LINKMODE, ROOTLINK | FAST, 0);
      else if ((*Name == 's') || (*Name == 'S'))
         trap8 (_LINKMODE, ROOTLINK | SLOW, 0);
      else
         return (ER_LINK_SYNTAX);
   }
   
   /* Reset link adaptor */
   trap8 (_RESETLNK, 0, 0);

   ActiveLink = 1;
   
   return (ActiveLink);
}


/* CloseLink --- close down the link connection */

int CloseLink (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   ActiveLink = NULL_LINK;

   return (SUCCEEDED);
}


/* ReadLink --- */

int ReadLink (LinkId, Buffer, Count, Timeout)
LINK LinkId;
char *Buffer;
unsigned int Count;
int Timeout;
{
   register unsigned int nb;

#ifdef TRACE
   printf ("ReadLink: %d bytes requested\n", Count);
#endif

   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (Count < 1)
      return (ER_LINK_CANT);

   if (SetTimeout (Timeout) == NO)
      return (ER_LINK_CANT);
   
   if (trap8 (_READ, Count, Buffer) == FAIL)
      nb = 0;     /* May not really be zero */
   else
      nb = Count;
      
#ifdef TRACE
   printf ("ReadLink: %d bytes read\n", nb);
#endif

   return ((int)nb);
}   


/* WriteLink --- */

int WriteLink (LinkId, Buffer, Count, Timeout)
LINK LinkId;
char *Buffer;
unsigned int Count;
int Timeout;
{
   register unsigned int nb;

#ifdef TRACE
   printf ("WriteLink: %d bytes requested\n", Count);
#endif

   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (Count < 1)
      return (ER_LINK_CANT);
   
   if (SetTimeout (Timeout) == NO)
      return (ER_LINK_CANT);
   
   if (trap8 (_WRITE, Count, Buffer) == FAIL)
      nb = 0;     /* May not really be zero */
   else
      nb = Count;
      
#ifdef TRACE
   printf ("WriteLink: %d bytes written\n", nb);
#endif

   return ((int)nb);
}


/* ResetLink --- */

int ResetLink (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);
   
   if (trap8 (_RESETNDS, 0, 0) == FAIL)
      return (ER_LINK_CANT);

   return (SUCCEEDED);
}


/* AnalyseLink --- */

int AnalyseLink (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (trap8 (_ANALNDS, 0, 0) == FAIL)
      return (ER_LINK_CANT);

   return (SUCCEEDED);
} 


/* TestError --- */

int TestError (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (trap8 (_TEST_ERR, 0, 0))
      return (1);
   
   return (0);
}  


/* TestRead --- */

int TestRead (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (trap8 (_RDRDY, 0, 0) == FAIL)
      return (0);
   else
      return (1);
}


/* TestWrite --- */

int TestWrite (LinkId)
LINK LinkId;
{
   if (LinkId != ActiveLink)
      return (ER_LINK_BAD);

   if (trap8 (_WRRDY, 0, 0) == FAIL)
      return (0);
   else
      return (1);
}


/* SetTimeout --- set the length of the read/write timeout */

static int SetTimeout (t)
int t;
{
   static int CurTimeout = -1;
   
   if (t != CurTimeout) {
      if (trap8x (_SETSYSVAR, _SETTO, t * 20, 0) == FAIL)
         return (NO);
         
      CurTimeout = t;
   }
   
   return (YES);
}


/* Installed --- return YES if the ACSILA driver is installed */

static int Installed ()
{
   unsigned long int vec8;
   unsigned long int *memtop = (unsigned long int *)0x0436;
   unsigned long int limit;
   void *sp;
   char *id;
   
   /* Read the trap 8 vector */
   vec8 = Setexc (TRAP8VEC, (void *)-1L);

   /* Read the top-of-memory adress */
   sp = Super ((void *)0L);
   limit = *memtop;
   Super (sp);
   
#ifdef DB
   printf ("vec8 = %08lx limit = %08lx\n", vec8, limit);
#endif

   /* Validate the trap 8 vector */
   if (vec8 == 0L || vec8 > limit)
      return (NO);
   
   /* There should be a pointer just before the trap 8 address */
   id = *((char **)(vec8 - 4L));
   
   /* Check that the pointer is valid */
   if (id == 0L || (unsigned long int)id > limit)
      return (NO);
   
#ifdef DB
   printf ("id = %08lx *id = %s\n", id, id);
#endif

   /* Look for the signature string */
   if (strncmp (id, "ACSILA", 6) != 0)
      return (NO);

   return (YES);
}
